/*
 * Decompiled with CFR 0.152.
 */
package org.jblas;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Arrays;
import org.jblas.ComplexDouble;
import org.jblas.DoubleMatrix;
import org.jblas.JavaBlas;
import org.jblas.NativeBlas;
import org.jblas.SimpleBlas;
import org.jblas.exceptions.SizeException;

public class ComplexDoubleMatrix {
    public int rows;
    public int columns;
    public int length;
    public double[] data = null;

    public ComplexDoubleMatrix(int newRows, int newColumns, double ... newData) {
        this.rows = newRows;
        this.columns = newColumns;
        this.length = this.rows * this.columns;
        if (newData.length != 2 * newRows * newColumns) {
            throw new IllegalArgumentException("Passed data must match matrix dimensions.");
        }
        this.data = newData;
    }

    public ComplexDoubleMatrix(int newRows, int newColumns) {
        this(newRows, newColumns, new double[2 * newRows * newColumns]);
    }

    public ComplexDoubleMatrix() {
        this(0, 0, null);
    }

    public ComplexDoubleMatrix(int len) {
        this(len, 1, new double[2 * len]);
    }

    public ComplexDoubleMatrix(double[] newData) {
        this(newData.length / 2, 1, newData);
    }

    public ComplexDoubleMatrix(ComplexDouble[] newData) {
        this(newData.length);
        for (int i = 0; i < newData.length; ++i) {
            this.put(i, newData[i]);
        }
    }

    public ComplexDoubleMatrix(DoubleMatrix m) {
        this(m.rows, m.columns);
        NativeBlas.dcopy(m.length, m.data, 0, 1, this.data, 0, 2);
    }

    public ComplexDoubleMatrix(DoubleMatrix real, DoubleMatrix imag) {
        this(real.rows, real.columns);
        real.assertSameSize(imag);
        if (real != null) {
            NativeBlas.dcopy(this.length, real.data, 0, 1, this.data, 0, 2);
        }
        if (imag != null) {
            NativeBlas.dcopy(this.length, imag.data, 0, 1, this.data, 1, 2);
        }
    }

    public ComplexDoubleMatrix(String filename) throws IOException {
        this.load(filename);
    }

    public ComplexDoubleMatrix(double[][] data) {
        this(data.length, data[0].length);
        int r;
        for (r = 0; r < this.rows; ++r) {
            assert (data[r].length == this.columns);
        }
        for (r = 0; r < this.rows; ++r) {
            for (int c = 0; c < this.columns; ++c) {
                this.put(r, c, data[r][c]);
            }
        }
    }

    public static ComplexDoubleMatrix zeros(int rows, int columns) {
        return new ComplexDoubleMatrix(rows, columns);
    }

    public static ComplexDoubleMatrix zeros(int length) {
        return ComplexDoubleMatrix.zeros(length, 1);
    }

    public static ComplexDoubleMatrix ones(int rows, int columns) {
        ComplexDoubleMatrix m = new ComplexDoubleMatrix(rows, columns);
        for (int i = 0; i < rows * columns; ++i) {
            m.put(i, 1.0);
        }
        return m;
    }

    public static ComplexDoubleMatrix ones(int length) {
        return ComplexDoubleMatrix.ones(length, 1);
    }

    public static ComplexDoubleMatrix diag(ComplexDoubleMatrix x) {
        ComplexDoubleMatrix m = new ComplexDoubleMatrix(x.length, x.length);
        for (int i = 0; i < x.length; ++i) {
            m.put(i, i, x.get(i));
        }
        return m;
    }

    public static ComplexDoubleMatrix diag(ComplexDoubleMatrix x, int rows, int columns) {
        if (x.length > rows || x.length > columns) {
            throw new SizeException("Length of diagonal matrix must be larger than both rows and columns.");
        }
        ComplexDoubleMatrix m = new ComplexDoubleMatrix(rows, columns);
        for (int i = 0; i < x.length; ++i) {
            m.put(i, i, x.get(i));
        }
        return m;
    }

    public static ComplexDoubleMatrix scalar(double s) {
        ComplexDoubleMatrix m = new ComplexDoubleMatrix(1, 1);
        m.put(0, 0, s);
        return m;
    }

    public boolean isScalar() {
        return this.length == 1;
    }

    public ComplexDouble scalar() {
        return this.get(0);
    }

    public static ComplexDoubleMatrix concatHorizontally(ComplexDoubleMatrix A, ComplexDoubleMatrix B) {
        if (A.rows != B.rows) {
            throw new SizeException("Matrices don't have same number of rows.");
        }
        ComplexDoubleMatrix result = new ComplexDoubleMatrix(A.rows, A.columns + B.columns);
        SimpleBlas.copy(A, result);
        NativeBlas.zcopy(B.length, B.data, 0, 1, result.data, A.length, 1);
        return result;
    }

    public static ComplexDoubleMatrix concatVertically(ComplexDoubleMatrix A, ComplexDoubleMatrix B) {
        if (A.columns != B.columns) {
            throw new SizeException("Matrices don't have same number of columns.");
        }
        ComplexDoubleMatrix result = new ComplexDoubleMatrix(A.rows + B.rows, A.columns);
        for (int i = 0; i < A.columns; ++i) {
            NativeBlas.zcopy(A.rows, A.data, A.index(0, i), 1, result.data, result.index(0, i), 1);
            NativeBlas.zcopy(B.rows, B.data, B.index(0, i), 1, result.data, result.index(A.rows, i), 1);
        }
        return result;
    }

    public ComplexDoubleMatrix get(int[] indices) {
        ComplexDoubleMatrix result = new ComplexDoubleMatrix(indices.length);
        for (int i = 0; i < indices.length; ++i) {
            result.put(i, this.get(indices[i]));
        }
        return result;
    }

    public ComplexDoubleMatrix get(int r, int[] indices) {
        ComplexDoubleMatrix result = new ComplexDoubleMatrix(1, indices.length);
        for (int i = 0; i < indices.length; ++i) {
            result.put(i, this.get(r, indices[i]));
        }
        return result;
    }

    public ComplexDoubleMatrix get(int[] indices, int c) {
        ComplexDoubleMatrix result = new ComplexDoubleMatrix(indices.length, 1);
        for (int i = 0; i < indices.length; ++i) {
            result.put(i, this.get(indices[i], c));
        }
        return result;
    }

    public ComplexDoubleMatrix get(int[] rindices, int[] cindices) {
        ComplexDoubleMatrix result = new ComplexDoubleMatrix(rindices.length, cindices.length);
        for (int i = 0; i < rindices.length; ++i) {
            for (int j = 0; j < cindices.length; ++j) {
                result.put(i, j, this.get(rindices[i], cindices[j]));
            }
        }
        return result;
    }

    public ComplexDoubleMatrix get(ComplexDoubleMatrix indices) {
        return this.get(indices.findIndices());
    }

    public ComplexDoubleMatrix get(int r, ComplexDoubleMatrix indices) {
        return this.get(r, indices.findIndices());
    }

    public ComplexDoubleMatrix get(ComplexDoubleMatrix indices, int c) {
        return this.get(indices.findIndices(), c);
    }

    public ComplexDoubleMatrix get(ComplexDoubleMatrix rindices, ComplexDoubleMatrix cindices) {
        return this.get(rindices.findIndices(), cindices.findIndices());
    }

    private void checkLength(int l) {
        if (this.length != l) {
            throw new SizeException("Matrix does not have the necessary length (" + this.length + " != " + l + ").");
        }
    }

    private void checkRows(int r) {
        if (this.rows != r) {
            throw new SizeException("Matrix does not have the necessary length (" + this.length + " != " + r + ").");
        }
    }

    private void checkColumns(int c) {
        if (this.columns != c) {
            throw new SizeException("Matrix does not have the necessary length (" + this.length + " != " + c + ").");
        }
    }

    public ComplexDoubleMatrix put(int[] indices, ComplexDoubleMatrix x) {
        if (x.isScalar()) {
            return this.put(indices, x.scalar());
        }
        x.checkLength(indices.length);
        for (int i = 0; i < indices.length; ++i) {
            this.put(indices[i], x.get(i));
        }
        return this;
    }

    public ComplexDoubleMatrix put(int r, int[] indices, ComplexDoubleMatrix x) {
        if (x.isScalar()) {
            return this.put(r, indices, x.scalar());
        }
        x.checkColumns(indices.length);
        for (int i = 0; i < indices.length; ++i) {
            this.put(r, indices[i], x.get(i));
        }
        return this;
    }

    public ComplexDoubleMatrix put(int[] indices, int c, ComplexDoubleMatrix x) {
        if (x.isScalar()) {
            return this.put(indices, c, x.scalar());
        }
        x.checkRows(indices.length);
        for (int i = 0; i < indices.length; ++i) {
            this.put(indices[i], c, x.get(i));
        }
        return this;
    }

    public ComplexDoubleMatrix put(int[] rindices, int[] cindices, ComplexDoubleMatrix x) {
        if (x.isScalar()) {
            return this.put(rindices, cindices, x.scalar());
        }
        x.checkRows(rindices.length);
        x.checkColumns(cindices.length);
        for (int i = 0; i < rindices.length; ++i) {
            for (int j = 0; j < cindices.length; ++j) {
                this.put(rindices[i], cindices[j], x.get(i, j));
            }
        }
        return this;
    }

    public ComplexDoubleMatrix put(int[] indices, double v) {
        for (int i = 0; i < indices.length; ++i) {
            this.put(indices[i], v);
        }
        return this;
    }

    public ComplexDoubleMatrix putReal(int[] indices, double v) {
        return this.put(indices, v);
    }

    public ComplexDoubleMatrix putImag(int[] indices, double v) {
        for (int i = 0; i < indices.length; ++i) {
            this.putImag(indices[i], v);
        }
        return this;
    }

    public ComplexDoubleMatrix put(int[] indices, ComplexDouble v) {
        for (int i = 0; i < indices.length; ++i) {
            this.put(indices[i], v);
        }
        return this;
    }

    public ComplexDoubleMatrix put(int r, int[] indices, double v) {
        for (int i = 0; i < indices.length; ++i) {
            this.put(r, indices[i], v);
        }
        return this;
    }

    public ComplexDoubleMatrix putReal(int r, int[] indices, double v) {
        return this.put(r, indices, v);
    }

    public ComplexDoubleMatrix putImag(int r, int[] indices, double v) {
        for (int i = 0; i < indices.length; ++i) {
            this.putImag(r, indices[i], v);
        }
        return this;
    }

    public ComplexDoubleMatrix put(int r, int[] indices, ComplexDouble v) {
        for (int i = 0; i < indices.length; ++i) {
            this.put(r, indices[i], v);
        }
        return this;
    }

    public ComplexDoubleMatrix put(int[] indices, int c, double v) {
        for (int i = 0; i < indices.length; ++i) {
            this.put(indices[i], c, v);
        }
        return this;
    }

    public ComplexDoubleMatrix putReal(int[] indices, int c, double v) {
        return this.put(indices, c, v);
    }

    public ComplexDoubleMatrix putImag(int[] indices, int c, double v) {
        for (int i = 0; i < indices.length; ++i) {
            this.putImag(indices[i], c, v);
        }
        return this;
    }

    public ComplexDoubleMatrix put(int[] indices, int c, ComplexDouble v) {
        for (int i = 0; i < indices.length; ++i) {
            this.put(indices[i], c, v);
        }
        return this;
    }

    public ComplexDoubleMatrix put(int[] rindices, int[] cindices, double v) {
        for (int i = 0; i < rindices.length; ++i) {
            for (int j = 0; j < cindices.length; ++j) {
                this.put(rindices[i], cindices[j], v);
            }
        }
        return this;
    }

    public ComplexDoubleMatrix putReal(int[] rindices, int[] cindices, double v) {
        return this.put(rindices, cindices, v);
    }

    public ComplexDoubleMatrix putImag(int[] rindices, int[] cindices, double v) {
        for (int i = 0; i < rindices.length; ++i) {
            for (int j = 0; j < cindices.length; ++j) {
                this.put(rindices[i], cindices[j], v);
            }
        }
        return this;
    }

    public ComplexDoubleMatrix put(int[] rindices, int[] cindices, ComplexDouble v) {
        for (int i = 0; i < rindices.length; ++i) {
            for (int j = 0; j < cindices.length; ++j) {
                this.put(rindices[i], cindices[j], v);
            }
        }
        return this;
    }

    public ComplexDoubleMatrix put(ComplexDoubleMatrix indices, ComplexDoubleMatrix v) {
        return this.put(indices.findIndices(), v);
    }

    public ComplexDoubleMatrix put(int r, ComplexDoubleMatrix indices, ComplexDoubleMatrix v) {
        return this.put(r, indices.findIndices(), v);
    }

    public ComplexDoubleMatrix put(ComplexDoubleMatrix indices, int c, ComplexDoubleMatrix v) {
        return this.put(indices.findIndices(), c, v);
    }

    public ComplexDoubleMatrix put(ComplexDoubleMatrix rindices, ComplexDoubleMatrix cindices, ComplexDoubleMatrix v) {
        return this.put(rindices.findIndices(), cindices.findIndices(), v);
    }

    public ComplexDoubleMatrix put(ComplexDoubleMatrix indices, double v) {
        return this.put(indices.findIndices(), v);
    }

    public ComplexDoubleMatrix putReal(ComplexDoubleMatrix indices, double v) {
        return this.put(indices, v);
    }

    public ComplexDoubleMatrix putImag(ComplexDoubleMatrix indices, double v) {
        return this.putImag(indices.findIndices(), v);
    }

    public ComplexDoubleMatrix put(ComplexDoubleMatrix indices, ComplexDouble v) {
        return this.put(indices.findIndices(), v);
    }

    public ComplexDoubleMatrix put(int r, ComplexDoubleMatrix indices, double v) {
        return this.put(r, indices.findIndices(), v);
    }

    public ComplexDoubleMatrix putReal(int r, ComplexDoubleMatrix indices, double v) {
        return this.put(r, indices, v);
    }

    public ComplexDoubleMatrix putImag(int r, ComplexDoubleMatrix indices, double v) {
        return this.putImag(r, indices.findIndices(), v);
    }

    public ComplexDoubleMatrix put(int r, ComplexDoubleMatrix indices, ComplexDouble v) {
        return this.put(r, indices.findIndices(), v);
    }

    public ComplexDoubleMatrix put(ComplexDoubleMatrix indices, int c, double v) {
        return this.put(indices.findIndices(), c, v);
    }

    public ComplexDoubleMatrix putReal(ComplexDoubleMatrix indices, int c, double v) {
        return this.put(indices, c, v);
    }

    public ComplexDoubleMatrix putImag(ComplexDoubleMatrix indices, int c, double v) {
        return this.putImag(indices.findIndices(), c, v);
    }

    public ComplexDoubleMatrix put(ComplexDoubleMatrix indices, int c, ComplexDouble v) {
        return this.put(indices.findIndices(), c, v);
    }

    public ComplexDoubleMatrix put(ComplexDoubleMatrix rindices, ComplexDoubleMatrix cindices, double v) {
        return this.put(rindices.findIndices(), cindices.findIndices(), v);
    }

    public ComplexDoubleMatrix putReal(ComplexDoubleMatrix rindices, ComplexDoubleMatrix cindices, double v) {
        return this.putReal(rindices.findIndices(), cindices.findIndices(), v);
    }

    public ComplexDoubleMatrix putImag(ComplexDoubleMatrix rindices, ComplexDoubleMatrix cindices, double v) {
        return this.putImag(rindices.findIndices(), cindices.findIndices(), v);
    }

    public ComplexDoubleMatrix put(ComplexDoubleMatrix rindices, ComplexDoubleMatrix cindices, ComplexDouble v) {
        return this.put(rindices.findIndices(), cindices.findIndices(), v);
    }

    public int[] findIndices() {
        int len = 0;
        for (int i = 0; i < this.length; ++i) {
            if (this.get(i).isZero()) continue;
            ++len;
        }
        int[] indices = new int[len];
        int c = 0;
        for (int i = 0; i < this.length; ++i) {
            if (this.get(i).isZero()) continue;
            indices[c++] = i;
        }
        return indices;
    }

    public ComplexDoubleMatrix transpose() {
        ComplexDoubleMatrix result = new ComplexDoubleMatrix(this.columns, this.rows);
        ComplexDouble c = new ComplexDouble(0.0);
        for (int i = 0; i < this.rows; ++i) {
            for (int j = 0; j < this.columns; ++j) {
                result.put(j, i, this.get(i, j, c));
            }
        }
        return result;
    }

    public ComplexDoubleMatrix hermitian() {
        ComplexDoubleMatrix result = new ComplexDoubleMatrix(this.columns, this.rows);
        ComplexDouble c = new ComplexDouble(0.0);
        for (int i = 0; i < this.rows; ++i) {
            for (int j = 0; j < this.columns; ++j) {
                result.put(j, i, this.get(i, j, c).conji());
            }
        }
        return result;
    }

    public ComplexDoubleMatrix conji() {
        ComplexDouble c = new ComplexDouble(0.0);
        for (int i = 0; i < this.length; ++i) {
            this.put(i, this.get(i, c).conji());
        }
        return this;
    }

    public ComplexDoubleMatrix conj() {
        return this.dup().conji();
    }

    public boolean equals(Object o) {
        if (!(o instanceof ComplexDoubleMatrix)) {
            return false;
        }
        ComplexDoubleMatrix other = (ComplexDoubleMatrix)o;
        if (!this.sameSize(other)) {
            return false;
        }
        return Arrays.equals(this.data, other.data);
    }

    public int hashCode() {
        return this.rows ^ this.columns ^ Arrays.hashCode(this.data);
    }

    public void resize(int newRows, int newColumns) {
        this.rows = newRows;
        this.columns = newColumns;
        this.length = newRows * newColumns;
        this.data = new double[2 * this.rows * this.columns];
    }

    public ComplexDoubleMatrix reshape(int newRows, int newColumns) {
        if (this.length != newRows * newColumns) {
            throw new IllegalArgumentException("Number of elements must not change.");
        }
        this.rows = newRows;
        this.columns = newColumns;
        return this;
    }

    public boolean sameSize(ComplexDoubleMatrix a) {
        return this.rows == a.rows && this.columns == a.columns;
    }

    public void assertSameSize(ComplexDoubleMatrix a) {
        if (!this.sameSize(a)) {
            throw new SizeException("Matrices must have the same size.");
        }
    }

    public boolean multipliesWith(ComplexDoubleMatrix a) {
        return this.columns == a.rows;
    }

    public void assertMultipliesWith(ComplexDoubleMatrix a) {
        if (!this.multipliesWith(a)) {
            throw new SizeException("Number of columns of left matrix must be equal to number of rows of right matrix.");
        }
    }

    public boolean sameLength(ComplexDoubleMatrix a) {
        return this.length == a.length;
    }

    public void assertSameLength(ComplexDoubleMatrix a) {
        if (!this.sameLength(a)) {
            throw new SizeException("Matrices must have same length (is: " + this.length + " and " + a.length + ")");
        }
    }

    public ComplexDoubleMatrix copy(ComplexDoubleMatrix a) {
        if (!this.sameSize(a)) {
            this.resize(a.rows, a.columns);
        }
        SimpleBlas.copy(a, this);
        return a;
    }

    public ComplexDoubleMatrix dup() {
        ComplexDoubleMatrix out = new ComplexDoubleMatrix(this.rows, this.columns);
        JavaBlas.rcopy(2 * this.length, this.data, 0, 1, out.data, 0, 1);
        return out;
    }

    public ComplexDoubleMatrix swapColumns(int i, int j) {
        NativeBlas.zswap(this.rows, this.data, this.index(0, i), 1, this.data, this.index(0, j), 1);
        return this;
    }

    public ComplexDoubleMatrix swapRows(int i, int j) {
        NativeBlas.zswap(this.columns, this.data, this.index(i, 0), this.rows, this.data, this.index(j, 0), this.rows);
        return this;
    }

    public ComplexDoubleMatrix put(int rowIndex, int columnIndex, double value) {
        this.data[2 * this.index((int)rowIndex, (int)columnIndex)] = value;
        return this;
    }

    public ComplexDoubleMatrix put(int rowIndex, int columnIndex, double realValue, double complexValue) {
        this.data[2 * this.index((int)rowIndex, (int)columnIndex)] = realValue;
        this.data[2 * this.index((int)rowIndex, (int)columnIndex) + 1] = complexValue;
        return this;
    }

    public ComplexDoubleMatrix put(int rowIndex, int columnIndex, ComplexDouble value) {
        int i = 2 * this.index(rowIndex, columnIndex);
        this.data[i] = value.real();
        this.data[i + 1] = value.imag();
        return this;
    }

    public ComplexDoubleMatrix putReal(int rowIndex, int columnIndex, double value) {
        this.data[2 * this.index((int)rowIndex, (int)columnIndex)] = value;
        return this;
    }

    public ComplexDoubleMatrix putImag(int rowIndex, int columnIndex, double value) {
        this.data[2 * this.index((int)rowIndex, (int)columnIndex) + 1] = value;
        return this;
    }

    public ComplexDouble get(int rowIndex, int columnIndex) {
        int i = 2 * this.index(rowIndex, columnIndex);
        return new ComplexDouble(this.data[i], this.data[i + 1]);
    }

    public ComplexDouble get(int rowIndex, int columnIndex, ComplexDouble result) {
        return this.get(this.index(rowIndex, columnIndex), result);
    }

    public DoubleMatrix getReal() {
        DoubleMatrix result = new DoubleMatrix(this.rows, this.columns);
        NativeBlas.dcopy(this.length, this.data, 0, 2, result.data, 0, 1);
        return result;
    }

    public int index(int rowIndex, int columnIndex) {
        return this.rows * columnIndex + rowIndex;
    }

    public int indexRows(int i) {
        return i - this.indexColumns(i) * this.rows;
    }

    public int indexColumns(int i) {
        return i / this.rows;
    }

    public ComplexDouble get(int i) {
        return new ComplexDouble(this.data[i * 2], this.data[i * 2 + 1]);
    }

    public ComplexDouble get(int i, ComplexDouble result) {
        return result.set(this.data[i * 2], this.data[i * 2 + 1]);
    }

    public double getReal(int i) {
        return this.data[2 * i];
    }

    public double getImag(int i) {
        return this.data[2 * i + 1];
    }

    public ComplexDoubleMatrix put(int i, double v) {
        this.data[2 * i] = v;
        return this;
    }

    public ComplexDoubleMatrix put(int i, double r, double c) {
        this.data[2 * i] = r;
        this.data[2 * i + 1] = c;
        return this;
    }

    public ComplexDoubleMatrix put(int i, ComplexDouble v) {
        this.data[2 * i] = v.real();
        this.data[2 * i + 1] = v.imag();
        return this;
    }

    public ComplexDoubleMatrix putReal(int i, double v) {
        return this.put(i, v);
    }

    public ComplexDoubleMatrix putImag(int i, double v) {
        this.data[2 * i + 1] = v;
        return this;
    }

    public int getRows() {
        return this.rows;
    }

    public int getColumns() {
        return this.columns;
    }

    public int getLength() {
        return this.length;
    }

    public boolean isEmpty() {
        return this.columns == 0 || this.rows == 0;
    }

    public boolean isSquare() {
        return this.columns == this.rows;
    }

    public void assertSquare() {
        if (!this.isSquare()) {
            throw new SizeException("Matrix must be square!");
        }
    }

    public boolean isVector() {
        return this.columns == 1 || this.rows == 1;
    }

    public boolean isRowVector() {
        return this.columns == 1;
    }

    public boolean isColumnVector() {
        return this.rows == 1;
    }

    public ComplexDoubleMatrix diag() {
        ComplexDoubleMatrix d = new ComplexDoubleMatrix(this.rows);
        NativeBlas.zcopy(this.rows, this.data, 0, this.rows + 1, d.data, 0, 1);
        return d;
    }

    public DoubleMatrix real() {
        DoubleMatrix result = new DoubleMatrix(this.rows, this.columns);
        NativeBlas.dcopy(this.length, this.data, 0, 2, result.data, 0, 1);
        return result;
    }

    public DoubleMatrix imag() {
        DoubleMatrix result = new DoubleMatrix(this.rows, this.columns);
        NativeBlas.dcopy(this.length, this.data, 1, 2, result.data, 0, 1);
        return result;
    }

    public void print() {
        System.out.println(this.toString());
    }

    public String toString() {
        StringBuilder s = new StringBuilder();
        s.append("[");
        for (int i = 0; i < this.rows; ++i) {
            for (int j = 0; j < this.columns; ++j) {
                s.append(this.get(i, j));
                if (j >= this.columns - 1) continue;
                s.append(", ");
            }
            if (i >= this.rows - 1) continue;
            s.append("; ");
        }
        s.append("]");
        return s.toString();
    }

    public double[] toDoubleArray() {
        return (double[])this.data.clone();
    }

    public ComplexDouble[] toArray() {
        ComplexDouble[] array = new ComplexDouble[this.length];
        for (int i = 0; i < this.length; ++i) {
            array[i] = this.get(i);
        }
        return array;
    }

    public ComplexDouble[][] toArray2() {
        ComplexDouble[][] array = new ComplexDouble[this.rows][this.columns];
        for (int r = 0; r < this.rows; ++r) {
            for (int c = 0; c < this.columns; ++c) {
                array[r][c] = this.get(r, c);
            }
        }
        return array;
    }

    public boolean[] toBooleanArray() {
        boolean[] array = new boolean[this.length];
        for (int i = 0; i < this.length; ++i) {
            array[i] = !this.get(i).isZero();
        }
        return array;
    }

    public boolean[][] toBooleanArray2() {
        boolean[][] array = new boolean[this.rows][this.columns];
        for (int r = 0; r < this.rows; ++r) {
            for (int c = 0; c < this.columns; ++c) {
                array[r][c] = !this.get(r, c).isZero();
            }
        }
        return array;
    }

    private void ensureResultLength(ComplexDoubleMatrix other, ComplexDoubleMatrix result) {
        if (!this.sameLength(result)) {
            if (result == this || result == other) {
                throw new SizeException("Cannot resize result matrix because it is used in-place.");
            }
            result.resize(this.rows, this.columns);
        }
    }

    public ComplexDoubleMatrix addi(ComplexDoubleMatrix other, ComplexDoubleMatrix result) {
        if (other.isScalar()) {
            return this.addi(other.scalar(), result);
        }
        this.assertSameLength(other);
        this.ensureResultLength(other, result);
        if (result == this) {
            SimpleBlas.axpy(ComplexDouble.UNIT, other, result);
        } else if (result == other) {
            SimpleBlas.axpy(ComplexDouble.UNIT, this, result);
        } else {
            SimpleBlas.copy(this, result);
            SimpleBlas.axpy(ComplexDouble.UNIT, other, result);
        }
        return result;
    }

    public ComplexDoubleMatrix addi(ComplexDouble v, ComplexDoubleMatrix result) {
        this.ensureResultLength(null, result);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, this.get(i).add(v));
        }
        return result;
    }

    public ComplexDoubleMatrix addi(double v, ComplexDoubleMatrix result) {
        return this.addi(new ComplexDouble(v), result);
    }

    public ComplexDoubleMatrix subi(ComplexDoubleMatrix other, ComplexDoubleMatrix result) {
        if (other.isScalar()) {
            return this.subi(other.scalar(), result);
        }
        this.assertSameLength(other);
        this.ensureResultLength(other, result);
        if (result == this) {
            SimpleBlas.axpy(ComplexDouble.NEG_UNIT, other, result);
        } else if (result == other) {
            SimpleBlas.scal(ComplexDouble.NEG_UNIT, result);
            SimpleBlas.axpy(ComplexDouble.UNIT, this, result);
        } else {
            SimpleBlas.copy(this, result);
            SimpleBlas.axpy(ComplexDouble.NEG_UNIT, other, result);
        }
        return result;
    }

    public ComplexDoubleMatrix subi(ComplexDouble v, ComplexDoubleMatrix result) {
        this.ensureResultLength(null, result);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, this.get(i).sub(v));
        }
        return result;
    }

    public ComplexDoubleMatrix subi(double v, ComplexDoubleMatrix result) {
        return this.subi(new ComplexDouble(v), result);
    }

    public ComplexDoubleMatrix rsubi(ComplexDoubleMatrix other, ComplexDoubleMatrix result) {
        return other.subi(this, result);
    }

    public ComplexDoubleMatrix rsubi(ComplexDouble a, ComplexDoubleMatrix result) {
        this.ensureResultLength(null, result);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, a.sub(this.get(i)));
        }
        return result;
    }

    public ComplexDoubleMatrix rsubi(double a, ComplexDoubleMatrix result) {
        return this.rsubi(new ComplexDouble(a), result);
    }

    public ComplexDoubleMatrix muli(ComplexDoubleMatrix other, ComplexDoubleMatrix result) {
        if (other.isScalar()) {
            return this.muli(other.scalar(), result);
        }
        this.assertSameLength(other);
        this.ensureResultLength(other, result);
        ComplexDouble c = new ComplexDouble(0.0);
        ComplexDouble d = new ComplexDouble(0.0);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, this.get(i, c).muli(other.get(i, d)));
        }
        return result;
    }

    public ComplexDoubleMatrix muli(ComplexDouble v, ComplexDoubleMatrix result) {
        this.ensureResultLength(null, result);
        ComplexDouble c = new ComplexDouble(0.0);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, this.get(i, c).muli(v));
        }
        return result;
    }

    public ComplexDoubleMatrix muli(double v, ComplexDoubleMatrix result) {
        return this.muli(new ComplexDouble(v), result);
    }

    public ComplexDoubleMatrix mmuli(ComplexDoubleMatrix other, ComplexDoubleMatrix result) {
        if (other.isScalar()) {
            return this.muli(other.scalar(), result);
        }
        this.assertMultipliesWith(other);
        if (result.rows != this.rows || result.columns != other.columns) {
            if (result != this && result != other) {
                result.resize(this.rows, other.columns);
            } else {
                throw new SizeException("Cannot resize result matrix because it is used in-place.");
            }
        }
        if (result == this || result == other) {
            ComplexDoubleMatrix temp = new ComplexDoubleMatrix(result.rows, result.columns);
            SimpleBlas.gemm(ComplexDouble.UNIT, this, other, ComplexDouble.ZERO, temp);
            SimpleBlas.copy(temp, result);
        } else {
            SimpleBlas.gemm(ComplexDouble.UNIT, this, other, ComplexDouble.ZERO, result);
        }
        return result;
    }

    public ComplexDoubleMatrix mmuli(ComplexDouble v, ComplexDoubleMatrix result) {
        return this.muli(v, result);
    }

    public ComplexDoubleMatrix mmuli(double v, ComplexDoubleMatrix result) {
        return this.muli(v, result);
    }

    public ComplexDoubleMatrix divi(ComplexDoubleMatrix other, ComplexDoubleMatrix result) {
        if (other.isScalar()) {
            return this.divi(other.scalar(), result);
        }
        this.assertSameLength(other);
        this.ensureResultLength(other, result);
        ComplexDouble c1 = new ComplexDouble(0.0);
        ComplexDouble c2 = new ComplexDouble(0.0);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, this.get(i, c1).divi(other.get(i, c2)));
        }
        return result;
    }

    public ComplexDoubleMatrix divi(ComplexDouble a, ComplexDoubleMatrix result) {
        this.ensureResultLength(null, result);
        ComplexDouble c = new ComplexDouble(0.0);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, this.get(i, c).divi(a));
        }
        return result;
    }

    public ComplexDoubleMatrix divi(double a, ComplexDoubleMatrix result) {
        return this.divi(new ComplexDouble(a), result);
    }

    public ComplexDoubleMatrix rdivi(ComplexDoubleMatrix other, ComplexDoubleMatrix result) {
        if (other.isScalar()) {
            return this.divi(other.scalar(), result);
        }
        this.assertSameLength(other);
        this.ensureResultLength(other, result);
        ComplexDouble c1 = new ComplexDouble(0.0);
        ComplexDouble c2 = new ComplexDouble(0.0);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, other.get(i, c1).divi(this.get(i, c2)));
        }
        return result;
    }

    public ComplexDoubleMatrix rdivi(ComplexDouble a, ComplexDoubleMatrix result) {
        this.ensureResultLength(null, result);
        ComplexDouble c1 = new ComplexDouble(0.0);
        ComplexDouble c2 = new ComplexDouble(0.0);
        for (int i = 0; i < this.length; ++i) {
            c1.copy(a);
            result.put(i, c1.divi(this.get(i, c2)));
        }
        return result;
    }

    public ComplexDoubleMatrix rdivi(double a, ComplexDoubleMatrix result) {
        return this.rdivi(new ComplexDouble(a), result);
    }

    public ComplexDoubleMatrix negi() {
        ComplexDouble c = new ComplexDouble(0.0);
        for (int i = 0; i < this.length; ++i) {
            this.put(i, this.get(i, c).negi());
        }
        return this;
    }

    public ComplexDoubleMatrix neg() {
        return this.dup().negi();
    }

    public ComplexDoubleMatrix noti() {
        ComplexDouble c = new ComplexDouble(0.0);
        for (int i = 0; i < this.length; ++i) {
            this.put(i, this.get(i, c).isZero() ? 1.0 : 0.0);
        }
        return this;
    }

    public ComplexDoubleMatrix not() {
        return this.dup().noti();
    }

    public ComplexDoubleMatrix truthi() {
        ComplexDouble c = new ComplexDouble(0.0);
        for (int i = 0; i < this.length; ++i) {
            this.put(i, this.get(i, c).isZero() ? 0.0 : 1.0);
        }
        return this;
    }

    public ComplexDoubleMatrix truth() {
        return this.dup().truthi();
    }

    public ComplexDoubleMatrix rankOneUpdate(ComplexDouble alpha, ComplexDoubleMatrix x, ComplexDoubleMatrix y) {
        if (this.rows != x.length) {
            throw new SizeException("Vector x has wrong length (" + x.length + " != " + this.rows + ").");
        }
        if (this.columns != y.length) {
            throw new SizeException("Vector y has wrong length (" + x.length + " != " + this.columns + ").");
        }
        SimpleBlas.gerc(alpha, x, y, this);
        return this;
    }

    public ComplexDoubleMatrix rankOneUpdate(double alpha, ComplexDoubleMatrix x, ComplexDoubleMatrix y) {
        return this.rankOneUpdate(new ComplexDouble(alpha), x, y);
    }

    public ComplexDoubleMatrix rankOneUpdate(double alpha, ComplexDoubleMatrix x) {
        return this.rankOneUpdate(new ComplexDouble(alpha), x, x);
    }

    public ComplexDoubleMatrix rankOneUpdate(ComplexDouble alpha, ComplexDoubleMatrix x) {
        return this.rankOneUpdate(alpha, x, x);
    }

    public ComplexDoubleMatrix rankOneUpdate(ComplexDoubleMatrix x) {
        return this.rankOneUpdate(1.0, x, x);
    }

    public ComplexDoubleMatrix rankOneUpdate(ComplexDoubleMatrix x, ComplexDoubleMatrix y) {
        return this.rankOneUpdate(1.0, x, y);
    }

    public ComplexDouble sum() {
        ComplexDouble s = new ComplexDouble(0.0);
        ComplexDouble c = new ComplexDouble(0.0);
        for (int i = 0; i < this.length; ++i) {
            s.addi(this.get(i, c));
        }
        return s;
    }

    public ComplexDouble mean() {
        return this.sum().div(this.length);
    }

    public ComplexDouble dotc(ComplexDoubleMatrix other) {
        return SimpleBlas.dotc(this, other);
    }

    public ComplexDouble dotu(ComplexDoubleMatrix other) {
        return SimpleBlas.dotu(this, other);
    }

    public double norm2() {
        return SimpleBlas.nrm2(this);
    }

    public double normmax() {
        int i = SimpleBlas.iamax(this);
        return this.get(i).abs();
    }

    public double norm1() {
        return SimpleBlas.asum(this);
    }

    public ComplexDoubleMatrix columnSums() {
        ComplexDoubleMatrix v = new ComplexDoubleMatrix(1, this.columns);
        for (int c = 0; c < this.columns; ++c) {
            v.put(c, this.getColumn(c).sum());
        }
        return v;
    }

    public ComplexDoubleMatrix columnMeans() {
        return this.columnSums().divi(this.rows);
    }

    public ComplexDoubleMatrix rowSums() {
        ComplexDoubleMatrix v = new ComplexDoubleMatrix(this.rows);
        for (int r = 0; r < this.rows; ++r) {
            v.put(r, this.getRow(r).sum());
        }
        return v;
    }

    public ComplexDoubleMatrix rowMeans() {
        return this.rowSums().divi(this.columns);
    }

    public ComplexDoubleMatrix getColumn(int c) {
        ComplexDoubleMatrix result = new ComplexDoubleMatrix(this.rows, 1);
        NativeBlas.zcopy(this.rows, this.data, this.index(0, c), 1, result.data, 0, 1);
        return result;
    }

    public void putColumn(int c, ComplexDoubleMatrix v) {
        NativeBlas.zcopy(this.rows, v.data, 0, 1, this.data, this.index(0, c), 1);
    }

    public ComplexDoubleMatrix getRow(int r) {
        ComplexDoubleMatrix result = new ComplexDoubleMatrix(1, this.columns);
        NativeBlas.zcopy(this.columns, this.data, this.index(r, 0), this.rows, result.data, 0, 1);
        return result;
    }

    public void putRow(int r, ComplexDoubleMatrix v) {
        NativeBlas.zcopy(this.columns, v.data, 0, 1, this.data, this.index(r, 0), this.rows);
    }

    public void addRowVector(ComplexDoubleMatrix x) {
        for (int r = 0; r < this.rows; ++r) {
            NativeBlas.zaxpy(this.columns, ComplexDouble.UNIT, x.data, 0, 1, this.data, this.index(r, 0), this.rows);
        }
    }

    public void addColumnVector(ComplexDoubleMatrix x) {
        for (int c = 0; c < this.columns; ++c) {
            NativeBlas.zaxpy(this.rows, ComplexDouble.UNIT, x.data, 0, 1, this.data, this.index(0, c), 1);
        }
    }

    public void subRowVector(ComplexDoubleMatrix x) {
        for (int r = 0; r < this.rows; ++r) {
            NativeBlas.zaxpy(this.columns, ComplexDouble.NEG_UNIT, x.data, 0, 1, this.data, this.index(r, 0), this.rows);
        }
    }

    public void subColumnVector(ComplexDoubleMatrix x) {
        for (int c = 0; c < this.columns; ++c) {
            NativeBlas.zaxpy(this.rows, ComplexDouble.NEG_UNIT, x.data, 0, 1, this.data, this.index(0, c), 1);
        }
    }

    public void out(DataOutputStream dos) throws IOException {
        dos.writeUTF("double");
        dos.writeInt(this.columns);
        dos.writeInt(this.rows);
        dos.writeInt(this.data.length);
        for (int i = 0; i < this.data.length; ++i) {
            dos.writeDouble(this.data[i]);
        }
    }

    public void in(DataInputStream dis) throws IOException {
        if (!dis.readUTF().equals("double")) {
            throw new IllegalStateException("The matrix in the specified file is not of the correct type!");
        }
        this.columns = dis.readInt();
        this.rows = dis.readInt();
        int MAX = dis.readInt();
        this.data = new double[MAX];
        for (int i = 0; i < MAX; ++i) {
            this.data[i] = dis.readDouble();
        }
    }

    public void save(String filename) throws IOException {
        DataOutputStream dos = new DataOutputStream(new FileOutputStream(filename, false));
        this.out(dos);
    }

    public void load(String filename) throws IOException {
        DataInputStream dis = new DataInputStream(new FileInputStream(filename));
        this.in(dis);
    }

    public ComplexDoubleMatrix addi(ComplexDoubleMatrix other) {
        return this.addi(other, this);
    }

    public ComplexDoubleMatrix add(ComplexDoubleMatrix other) {
        return this.addi(other, new ComplexDoubleMatrix(this.rows, this.columns));
    }

    public ComplexDoubleMatrix addi(ComplexDouble v) {
        return this.addi(v, this);
    }

    public ComplexDoubleMatrix addi(double v) {
        return this.addi(new ComplexDouble(v), this);
    }

    public ComplexDoubleMatrix add(ComplexDouble v) {
        return this.addi(v, new ComplexDoubleMatrix(this.rows, this.columns));
    }

    public ComplexDoubleMatrix add(double v) {
        return this.addi(new ComplexDouble(v), new ComplexDoubleMatrix(this.rows, this.columns));
    }

    public ComplexDoubleMatrix subi(ComplexDoubleMatrix other) {
        return this.subi(other, this);
    }

    public ComplexDoubleMatrix sub(ComplexDoubleMatrix other) {
        return this.subi(other, new ComplexDoubleMatrix(this.rows, this.columns));
    }

    public ComplexDoubleMatrix subi(ComplexDouble v) {
        return this.subi(v, this);
    }

    public ComplexDoubleMatrix subi(double v) {
        return this.subi(new ComplexDouble(v), this);
    }

    public ComplexDoubleMatrix sub(ComplexDouble v) {
        return this.subi(v, new ComplexDoubleMatrix(this.rows, this.columns));
    }

    public ComplexDoubleMatrix sub(double v) {
        return this.subi(new ComplexDouble(v), new ComplexDoubleMatrix(this.rows, this.columns));
    }

    public ComplexDoubleMatrix rsubi(ComplexDoubleMatrix other) {
        return this.rsubi(other, this);
    }

    public ComplexDoubleMatrix rsub(ComplexDoubleMatrix other) {
        return this.rsubi(other, new ComplexDoubleMatrix(this.rows, this.columns));
    }

    public ComplexDoubleMatrix rsubi(ComplexDouble v) {
        return this.rsubi(v, this);
    }

    public ComplexDoubleMatrix rsubi(double v) {
        return this.rsubi(new ComplexDouble(v), this);
    }

    public ComplexDoubleMatrix rsub(ComplexDouble v) {
        return this.rsubi(v, new ComplexDoubleMatrix(this.rows, this.columns));
    }

    public ComplexDoubleMatrix rsub(double v) {
        return this.rsubi(new ComplexDouble(v), new ComplexDoubleMatrix(this.rows, this.columns));
    }

    public ComplexDoubleMatrix divi(ComplexDoubleMatrix other) {
        return this.divi(other, this);
    }

    public ComplexDoubleMatrix div(ComplexDoubleMatrix other) {
        return this.divi(other, new ComplexDoubleMatrix(this.rows, this.columns));
    }

    public ComplexDoubleMatrix divi(ComplexDouble v) {
        return this.divi(v, this);
    }

    public ComplexDoubleMatrix divi(double v) {
        return this.divi(new ComplexDouble(v), this);
    }

    public ComplexDoubleMatrix div(ComplexDouble v) {
        return this.divi(v, new ComplexDoubleMatrix(this.rows, this.columns));
    }

    public ComplexDoubleMatrix div(double v) {
        return this.divi(new ComplexDouble(v), new ComplexDoubleMatrix(this.rows, this.columns));
    }

    public ComplexDoubleMatrix rdivi(ComplexDoubleMatrix other) {
        return this.rdivi(other, this);
    }

    public ComplexDoubleMatrix rdiv(ComplexDoubleMatrix other) {
        return this.rdivi(other, new ComplexDoubleMatrix(this.rows, this.columns));
    }

    public ComplexDoubleMatrix rdivi(ComplexDouble v) {
        return this.rdivi(v, this);
    }

    public ComplexDoubleMatrix rdivi(double v) {
        return this.rdivi(new ComplexDouble(v), this);
    }

    public ComplexDoubleMatrix rdiv(ComplexDouble v) {
        return this.rdivi(v, new ComplexDoubleMatrix(this.rows, this.columns));
    }

    public ComplexDoubleMatrix rdiv(double v) {
        return this.rdivi(new ComplexDouble(v), new ComplexDoubleMatrix(this.rows, this.columns));
    }

    public ComplexDoubleMatrix muli(ComplexDoubleMatrix other) {
        return this.muli(other, this);
    }

    public ComplexDoubleMatrix mul(ComplexDoubleMatrix other) {
        return this.muli(other, new ComplexDoubleMatrix(this.rows, this.columns));
    }

    public ComplexDoubleMatrix muli(ComplexDouble v) {
        return this.muli(v, this);
    }

    public ComplexDoubleMatrix muli(double v) {
        return this.muli(new ComplexDouble(v), this);
    }

    public ComplexDoubleMatrix mul(ComplexDouble v) {
        return this.muli(v, new ComplexDoubleMatrix(this.rows, this.columns));
    }

    public ComplexDoubleMatrix mul(double v) {
        return this.muli(new ComplexDouble(v), new ComplexDoubleMatrix(this.rows, this.columns));
    }

    public ComplexDoubleMatrix mmuli(ComplexDoubleMatrix other) {
        return this.mmuli(other, this);
    }

    public ComplexDoubleMatrix mmul(ComplexDoubleMatrix other) {
        return this.mmuli(other, new ComplexDoubleMatrix(this.rows, other.columns));
    }

    public ComplexDoubleMatrix mmuli(ComplexDouble v) {
        return this.mmuli(v, this);
    }

    public ComplexDoubleMatrix mmuli(double v) {
        return this.mmuli(new ComplexDouble(v), this);
    }

    public ComplexDoubleMatrix mmul(ComplexDouble v) {
        return this.mmuli(v, new ComplexDoubleMatrix(this.rows, this.columns));
    }

    public ComplexDoubleMatrix mmul(double v) {
        return this.mmuli(new ComplexDouble(v), new ComplexDoubleMatrix(this.rows, this.columns));
    }

    public ComplexDoubleMatrix eqi(ComplexDoubleMatrix other, ComplexDoubleMatrix result) {
        if (other.isScalar()) {
            return this.eqi(other.scalar(), result);
        }
        this.assertSameLength(other);
        this.ensureResultLength(other, result);
        ComplexDouble c1 = new ComplexDouble(0.0);
        ComplexDouble c2 = new ComplexDouble(0.0);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, this.get(i, c1).eq(other.get(i, c2)) ? 1.0 : 0.0);
        }
        return result;
    }

    public ComplexDoubleMatrix eqi(ComplexDoubleMatrix other) {
        return this.eqi(other, this);
    }

    public ComplexDoubleMatrix eq(ComplexDoubleMatrix other) {
        return this.eqi(other, new ComplexDoubleMatrix(this.rows, this.columns));
    }

    public ComplexDoubleMatrix eqi(ComplexDouble value, ComplexDoubleMatrix result) {
        this.ensureResultLength(null, result);
        ComplexDouble c = new ComplexDouble(0.0);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, this.get(i, c).eq(value) ? 1.0 : 0.0);
        }
        return result;
    }

    public ComplexDoubleMatrix eqi(double value, ComplexDoubleMatrix result) {
        return this.eqi(new ComplexDouble(value), result);
    }

    public ComplexDoubleMatrix eqi(ComplexDouble value) {
        return this.eqi(value, this);
    }

    public ComplexDoubleMatrix eqi(double value) {
        return this.eqi(new ComplexDouble(value));
    }

    public ComplexDoubleMatrix eq(ComplexDouble value) {
        return this.eqi(value, new ComplexDoubleMatrix(this.rows, this.columns));
    }

    public ComplexDoubleMatrix eq(double value) {
        return this.eqi(new ComplexDouble(value));
    }

    public ComplexDoubleMatrix nei(ComplexDoubleMatrix other, ComplexDoubleMatrix result) {
        if (other.isScalar()) {
            return this.nei(other.scalar(), result);
        }
        this.assertSameLength(other);
        this.ensureResultLength(other, result);
        ComplexDouble c1 = new ComplexDouble(0.0);
        ComplexDouble c2 = new ComplexDouble(0.0);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, this.get(i, c1).eq(other.get(i, c2)) ? 1.0 : 0.0);
        }
        return result;
    }

    public ComplexDoubleMatrix nei(ComplexDoubleMatrix other) {
        return this.nei(other, this);
    }

    public ComplexDoubleMatrix ne(ComplexDoubleMatrix other) {
        return this.nei(other, new ComplexDoubleMatrix(this.rows, this.columns));
    }

    public ComplexDoubleMatrix nei(ComplexDouble value, ComplexDoubleMatrix result) {
        this.ensureResultLength(null, result);
        ComplexDouble c = new ComplexDouble(0.0);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, this.get(i, c).eq(value) ? 1.0 : 0.0);
        }
        return result;
    }

    public ComplexDoubleMatrix nei(double value, ComplexDoubleMatrix result) {
        return this.nei(new ComplexDouble(value), result);
    }

    public ComplexDoubleMatrix nei(ComplexDouble value) {
        return this.nei(value, this);
    }

    public ComplexDoubleMatrix nei(double value) {
        return this.nei(new ComplexDouble(value));
    }

    public ComplexDoubleMatrix ne(ComplexDouble value) {
        return this.nei(value, new ComplexDoubleMatrix(this.rows, this.columns));
    }

    public ComplexDoubleMatrix ne(double value) {
        return this.nei(new ComplexDouble(value));
    }

    public ComplexDoubleMatrix andi(ComplexDoubleMatrix other, ComplexDoubleMatrix result) {
        this.assertSameLength(other);
        this.ensureResultLength(other, result);
        ComplexDouble t1 = new ComplexDouble(0.0);
        ComplexDouble t2 = new ComplexDouble(0.0);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, !this.get(i, t1).isZero() & !other.get(i, t2).isZero() ? 1.0 : 0.0);
        }
        return result;
    }

    public ComplexDoubleMatrix andi(ComplexDoubleMatrix other) {
        return this.andi(other, this);
    }

    public ComplexDoubleMatrix and(ComplexDoubleMatrix other) {
        return this.andi(other, new ComplexDoubleMatrix(this.rows, this.columns));
    }

    public ComplexDoubleMatrix andi(ComplexDouble value, ComplexDoubleMatrix result) {
        this.ensureResultLength(null, result);
        boolean val = !value.isZero();
        ComplexDouble t = new ComplexDouble(0.0);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, !this.get(i, t).isZero() & val ? 1.0 : 0.0);
        }
        return result;
    }

    public ComplexDoubleMatrix andi(double value, ComplexDoubleMatrix result) {
        return this.andi(new ComplexDouble(value), result);
    }

    public ComplexDoubleMatrix andi(ComplexDouble value) {
        return this.andi(value, this);
    }

    public ComplexDoubleMatrix andi(double value) {
        return this.andi(new ComplexDouble(value), this);
    }

    public ComplexDoubleMatrix and(ComplexDouble value) {
        return this.andi(value, new ComplexDoubleMatrix(this.rows, this.columns));
    }

    public ComplexDoubleMatrix and(double value) {
        return this.andi(new ComplexDouble(value));
    }

    public ComplexDoubleMatrix ori(ComplexDoubleMatrix other, ComplexDoubleMatrix result) {
        this.assertSameLength(other);
        this.ensureResultLength(other, result);
        ComplexDouble t1 = new ComplexDouble(0.0);
        ComplexDouble t2 = new ComplexDouble(0.0);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, !this.get(i, t1).isZero() | !other.get(i, t2).isZero() ? 1.0 : 0.0);
        }
        return result;
    }

    public ComplexDoubleMatrix ori(ComplexDoubleMatrix other) {
        return this.ori(other, this);
    }

    public ComplexDoubleMatrix or(ComplexDoubleMatrix other) {
        return this.ori(other, new ComplexDoubleMatrix(this.rows, this.columns));
    }

    public ComplexDoubleMatrix ori(ComplexDouble value, ComplexDoubleMatrix result) {
        this.ensureResultLength(null, result);
        boolean val = !value.isZero();
        ComplexDouble t = new ComplexDouble(0.0);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, !this.get(i, t).isZero() | val ? 1.0 : 0.0);
        }
        return result;
    }

    public ComplexDoubleMatrix ori(double value, ComplexDoubleMatrix result) {
        return this.ori(new ComplexDouble(value), result);
    }

    public ComplexDoubleMatrix ori(ComplexDouble value) {
        return this.ori(value, this);
    }

    public ComplexDoubleMatrix ori(double value) {
        return this.ori(new ComplexDouble(value), this);
    }

    public ComplexDoubleMatrix or(ComplexDouble value) {
        return this.ori(value, new ComplexDoubleMatrix(this.rows, this.columns));
    }

    public ComplexDoubleMatrix or(double value) {
        return this.ori(new ComplexDouble(value));
    }

    public ComplexDoubleMatrix xori(ComplexDoubleMatrix other, ComplexDoubleMatrix result) {
        this.assertSameLength(other);
        this.ensureResultLength(other, result);
        ComplexDouble t1 = new ComplexDouble(0.0);
        ComplexDouble t2 = new ComplexDouble(0.0);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, !this.get(i, t1).isZero() ^ !other.get(i, t2).isZero() ? 1.0 : 0.0);
        }
        return result;
    }

    public ComplexDoubleMatrix xori(ComplexDoubleMatrix other) {
        return this.xori(other, this);
    }

    public ComplexDoubleMatrix xor(ComplexDoubleMatrix other) {
        return this.xori(other, new ComplexDoubleMatrix(this.rows, this.columns));
    }

    public ComplexDoubleMatrix xori(ComplexDouble value, ComplexDoubleMatrix result) {
        this.ensureResultLength(null, result);
        boolean val = !value.isZero();
        ComplexDouble t = new ComplexDouble(0.0);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, !this.get(i, t).isZero() ^ val ? 1.0 : 0.0);
        }
        return result;
    }

    public ComplexDoubleMatrix xori(double value, ComplexDoubleMatrix result) {
        return this.xori(new ComplexDouble(value), result);
    }

    public ComplexDoubleMatrix xori(ComplexDouble value) {
        return this.xori(value, this);
    }

    public ComplexDoubleMatrix xori(double value) {
        return this.xori(new ComplexDouble(value), this);
    }

    public ComplexDoubleMatrix xor(ComplexDouble value) {
        return this.xori(value, new ComplexDoubleMatrix(this.rows, this.columns));
    }

    public ComplexDoubleMatrix xor(double value) {
        return this.xori(new ComplexDouble(value));
    }
}

