/*
 * Decompiled with CFR 0.152.
 */
package neurord.numeric.grid;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import neurord.numeric.chem.ReactionTable;
import neurord.numeric.chem.StimulationTable;
import neurord.numeric.grid.AdaptiveGridCalc;
import neurord.numeric.grid.IGridCalc;
import neurord.numeric.math.MersenneTwister;
import neurord.numeric.math.RandomGenerator;
import neurord.numeric.morph.VolumeElement;
import neurord.numeric.morph.VolumeGrid;
import neurord.numeric.stochastic.StepGenerator;
import neurord.util.ArrayUtil;
import neurord.util.Logging;
import neurord.util.Settings;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class NextEventQueue {
    public static final Logger log = LogManager.getLogger();
    static final boolean update_times = Settings.getProperty("neurord.neq.update_times", "Update putative times using Gibson&Bruck", true);
    static final boolean only_init = Settings.getProperty("neurord.neq.only_init", "Terminate after initialization is finished", false);
    static final boolean check_updates = Settings.getProperty("neurord.neq.check_updates", "Perform extensive checks of propensity changes", false);
    static final boolean log_queue = Settings.getProperty("neurord.neq.log_queue", "Log debug info about queue operations", false);
    static final boolean log_propensity = Settings.getProperty("neurord.neq.log_propensity", "Log debug info about propensity calculations", false);
    static final boolean log_reposition = Settings.getProperty("neurord.neq.log_reposition", "Log debug info about movements in the queue", false);
    static final double log_debug_start = Settings.getProperty("neurord.neq.log_debug_start", "Turn on debugging at this time in simulation", Double.NaN);
    static final int log_start_events = Settings.getProperty("neurord.neq.log_start_events", "Print information about this many events at startup", 99);
    public static final int[] PLUS_ONE = new int[]{1};
    public static final int[] MINUS_ONE = new int[]{-1};
    public static final int[] MINUS_ONE_PLUS_ONE = new int[]{-1, 1};
    long leaps = 0L;
    long leap_extent = 0L;
    long normal_waits = 0L;
    final RandomGenerator random;
    final StepGenerator stepper;
    final int[][] particles;
    final double tolerance;
    boolean adaptive;
    final double leap_min_jump;
    final PriorityTree<NextEvent> queue = new PriorityTree();
    final HashMap<Integer, IndexDescription> stat_descriptions = new HashMap();
    private static boolean _warned_empty = false;

    static void log_dependency_edges(ArrayList<NextEvent> events) {
        int all = 0;
        int active = 0;
        for (NextEvent ev : events) {
            all += ev.dependent.size();
            if (!(ev.propensity > 0.0)) continue;
            active += ev.dependent.size();
        }
        log.info("{} dependency edges, {} active", all, active);
    }

    public static int[][] stoichiometry(int[] ri, int[] rs, int[] pi, int[] ps) {
        ArrayList<Integer> si = new ArrayList<Integer>();
        ArrayList<Integer> ss = new ArrayList<Integer>();
        boolean[] pconsidered = new boolean[pi.length];
        for (int i = 0; i < ri.length; ++i) {
            int j;
            for (j = 0; j < pi.length; ++j) {
                if (ri[i] != pi[j]) continue;
                pconsidered[j] = true;
                break;
            }
            if (j == pi.length) {
                si.add(ri[i]);
                ss.add(-rs[i]);
                continue;
            }
            if (rs[i] == ps[j]) continue;
            assert (ri[i] == pi[j]);
            si.add(ri[i]);
            ss.add(ps[j] - rs[i]);
        }
        for (int j = 0; j < pi.length; ++j) {
            if (pconsidered[j]) continue;
            si.add(pi[j]);
            ss.add(ps[j]);
        }
        return new int[][]{ArrayUtil.toArray(si), ArrayUtil.toArray(ss)};
    }

    public int updatePopulation(int element, int specie, int count, NextEvent event) {
        int done;
        if (count < 0 && this.particles[element][specie] < -count) {
            log.debug("{}: population would become negative for element {} sp {}: changing {} by {}", event, element, specie, this.particles[element][specie], count);
            done = -this.particles[element][specie];
            this.particles[element][specie] = 0;
        } else {
            int[] nArray = this.particles[element];
            int n = specie;
            nArray[n] = nArray[n] + count;
            done = count;
        }
        return done;
    }

    public NextEventQueue(RandomGenerator random, StepGenerator stepper, int[][] particles, boolean adaptive, double tolerance, double leap_min_jump) {
        this.random = random != null ? random : new MersenneTwister();
        this.stepper = stepper != null ? stepper : new StepGenerator(this.random);
        this.particles = particles;
        assert (0.0 <= tolerance && tolerance <= 1.0) : tolerance;
        this.tolerance = tolerance;
        this.adaptive = adaptive;
        this.leap_min_jump = leap_min_jump;
        if (this.adaptive) {
            log.info("Using {} as leap tolerance, jumping when {} times longer", tolerance, leap_min_jump);
        } else {
            log.info("Leaping disabled");
        }
        if (log_propensity && !check_updates) {
            log.warn("neurord.neq.log_propensity has no effect without neurord.neq.check_updates");
        }
    }

    IndexDescription makeIndex(String choice, HashMap<Integer, IndexDescription> map, Numbering numbering, IndexOption ... options) {
        for (IndexOption opt : options) {
            if (!opt.label.equals(choice)) continue;
            if (!map.containsKey(opt.ident)) {
                map.put(opt.ident, new IndexDescription(numbering.get()));
            }
            IndexDescription desc = map.get(opt.ident);
            return desc;
        }
        return null;
    }

    private static long neighboursToIndex(int el, int el2, int species, int nel, int nspecies) {
        return ((long)el * (long)nel + (long)el2) * (long)nspecies + (long)species;
    }

    ArrayList<NextDiffusion> createDiffusions(Numbering numbering, VolumeGrid grid, ReactionTable rtab, String statistics, Numbering stat_numbering) {
        double[] volumes = grid.getElementVolumes();
        int[][] neighbors = grid.getPerElementNeighbors();
        double[][] couplings = grid.getPerElementCouplingConstants();
        double[] fdiff = rtab.getDiffusionConstants();
        String[] species = rtab.getSpecies();
        ArrayList<NextDiffusion> ans = new ArrayList<NextDiffusion>(5 * neighbors.length);
        int nel = grid.size();
        HashMap<Long, NextDiffusion> rev = new HashMap<Long, NextDiffusion>();
        HashMap<Integer, IndexDescription> stat_indices = new HashMap<Integer, IndexDescription>();
        for (int el = 0; el < neighbors.length; ++el) {
            log.debug("el.{} neighbors {}", el, neighbors[el]);
            for (int j = 0; j < neighbors[el].length; ++j) {
                int el2 = neighbors[el][j];
                double cc = couplings[el][j];
                if (!(cc > 0.0)) continue;
                for (int sp = 0; sp < fdiff.length; ++sp) {
                    if (!(fdiff[sp] > 0.0)) continue;
                    int event_number = numbering.get();
                    IndexDescription stat_index = this.makeIndex(statistics, stat_indices, stat_numbering, new IndexOption("by-channel", sp), new IndexOption("by-event", event_number));
                    NextDiffusion diff = new NextDiffusion(event_number, stat_index, el, el2, j, sp, species[sp], fdiff[sp] * cc / volumes[el]);
                    ans.add(diff);
                    log.debug("diff {} \u2192 {}", grid.getElement(el).getNumber(), grid.getElement(el2).getNumber());
                    long revnumber = NextEventQueue.neighboursToIndex(el2, el, sp, nel, species.length);
                    NextDiffusion revdiff = (NextDiffusion)rev.get(revnumber);
                    if (revdiff != null) {
                        diff.addReverse(revdiff);
                    } else {
                        revnumber = NextEventQueue.neighboursToIndex(el, el2, sp, nel, species.length);
                        rev.put(revnumber, diff);
                    }
                    if (statistics.equals("by-channel")) {
                        stat_index.setDescription("Diffusion of " + species[sp]);
                        continue;
                    }
                    if (!statistics.equals("by-event")) continue;
                    stat_index.setDescription(diff.toString());
                }
            }
        }
        log.info("Created {} diffusion events", ans.size());
        return ans;
    }

    ArrayList<NextReaction> createReactions(Numbering numbering, VolumeGrid grid, ReactionTable rtab, String statistics, Numbering stat_numbering) {
        int r;
        double[] volumes = grid.getElementVolumes();
        int n = rtab.getNReaction() * volumes.length;
        int[][] RI = rtab.getReactantIndices();
        int[][] PI = rtab.getProductIndices();
        int[][] RS = rtab.getReactantStoichiometry();
        int[][] PS = rtab.getProductStoichiometry();
        int[][] RP = rtab.getReactantPowers();
        int[] reversible_pairs = rtab.getReversiblePairs();
        log.debug("reversible_pairs: {}", new Object[]{reversible_pairs});
        String[] species = rtab.getSpecies();
        HashMap<Integer, IndexDescription> stat_indices = new HashMap<Integer, IndexDescription>();
        ArrayList<NextReaction> ans = new ArrayList<NextReaction>(RI.length * volumes.length);
        for (r = 0; r < rtab.getNReaction(); ++r) {
            int[] ri = RI[r];
            int[] pi = PI[r];
            int[] rs = RS[r];
            int[] ps = PS[r];
            int[] rp = RP[r];
            double rate = rtab.getRates()[r];
            for (int el = 0; el < volumes.length; ++el) {
                String signature = ReactionTable.getReactionSignature(ri, rs, pi, ps, species);
                int event_number = numbering.get();
                IndexDescription stat_index = this.makeIndex(statistics, stat_indices, stat_numbering, new IndexOption("by-channel", r), new IndexOption("by-event", event_number));
                NextReaction ev = new NextReaction(event_number, stat_index, r, el, ri, pi, rs, ps, rp, signature, rate, volumes[el]);
                ans.add(ev);
                if (statistics.equals("by-channel")) {
                    stat_index.setDescription("Reaction " + signature);
                    continue;
                }
                if (!statistics.equals("by-event")) continue;
                stat_index.setDescription(ev.toString());
            }
        }
        for (r = 0; r < rtab.getNReaction(); ++r) {
            if (reversible_pairs[r] < 0) continue;
            for (int el = 0; el < volumes.length; ++el) {
                NextReaction one = ans.get(r * volumes.length + el);
                NextReaction two = ans.get(reversible_pairs[r] * volumes.length + el);
                one.addReverse(two);
            }
        }
        log.info("Created {} reaction events", ans.size());
        return ans;
    }

    ArrayList<NextStimulation> createStimulations(Numbering numbering, VolumeGrid grid, ReactionTable rtab, StimulationTable stimtab, String statistics, Numbering stat_numbering) {
        String[] species = rtab.getSpecies();
        HashMap<Integer, IndexDescription> stat_indices = new HashMap<Integer, IndexDescription>();
        ArrayList<StimulationTable.Stimulation> stims = stimtab.getStimulations();
        ArrayList<NextStimulation> ans = new ArrayList<NextStimulation>();
        for (int n = 0; n < stims.size(); ++n) {
            StimulationTable.Stimulation stim = stims.get(n);
            ArrayList<VolumeElement> targets = grid.filterElementsByLabel(stim.site);
            boolean submembrane = stim.site.endsWith(":submembrane");
            double sum = 0.0;
            for (VolumeElement el : targets) {
                sum += el.getExposedArea();
            }
            for (VolumeElement el : targets) {
                int event_number = numbering.get();
                IndexDescription stat_index = this.makeIndex(statistics, stat_indices, stat_numbering, new IndexOption("by-channel", n), new IndexOption("by-event", event_number), new IndexOption("injections", stim.species));
                double fraction = submembrane ? el.getExposedArea() / sum : 1.0 / (double)targets.size();
                NextStimulation ev = new NextStimulation(event_number, stat_index, el.getNumber(), fraction, stim.species, species[stim.species], stim);
                ans.add(ev);
                if (statistics.equals("by-channel")) {
                    stat_index.setDescription(String.format("Stimulation %s\u2192%s", species[stim.species], stim.site));
                    continue;
                }
                if (statistics.equals("by-event")) {
                    stat_index.setDescription(ev.toString());
                    continue;
                }
                if (!statistics.equals("injections")) continue;
                stat_index.setDescription("Stimulation of species " + species[stim.species]);
            }
        }
        log.info("Created {} stimulation events", ans.size());
        return ans;
    }

    public int stat_count(String statistics, String[] species) {
        HashSet<Integer> set = new HashSet<Integer>();
        for (IGridCalc.Event ev : this.getEvents()) {
            int stat_index = ev.stat_index();
            if (stat_index < 0) continue;
            set.add(stat_index);
        }
        switch (statistics) {
            case "injections": 
            case "by-channel": {
                return set.size();
            }
            case "by-event": {
                assert (set.size() == this.getEvents().size());
                return this.getEvents().size();
            }
        }
        return -1;
    }

    public static NextEventQueue create(int[][] particles, RandomGenerator random, StepGenerator stepper, VolumeGrid grid, ReactionTable rtab, StimulationTable stimtab, boolean adaptive, double tolerance, double leap_min_jump, boolean verbose, String statistics) {
        NextEventQueue obj = new NextEventQueue(random, stepper, particles, adaptive, tolerance, leap_min_jump);
        ArrayList<NextEvent> e = new ArrayList<NextEvent>();
        Numbering numbering = new Numbering();
        Numbering stat_numbering = new Numbering();
        e.addAll(obj.createDiffusions(numbering, grid, rtab, statistics, stat_numbering));
        e.addAll(obj.createReactions(numbering, grid, rtab, statistics, stat_numbering));
        e.addAll(obj.createStimulations(numbering, grid, rtab, stimtab, statistics, stat_numbering));
        obj.queue.build(e.toArray(new NextEvent[0]));
        log.debug("Creating dependency graph");
        HashMap<Integer, ArrayList<NextEvent>> map = new HashMap<Integer, ArrayList<NextEvent>>();
        for (NextEvent ev : e) {
            if (verbose) {
                log.debug("{}:{}", ev.index(), ev);
            }
            map.putIfAbsent(ev.element(), new ArrayList());
            ((ArrayList)map.get(ev.element())).add(ev);
        }
        int i = 0;
        for (NextEvent ev : e) {
            ev.addRelations(map, rtab.getSpecies(), verbose && (log_start_events == -1 || i++ < log_start_events));
        }
        if (verbose) {
            log.info("{} event channels:", ((NextEvent[])obj.queue.nodes).length);
            i = 0;
            for (NextEvent ev : (NextEvent[])obj.queue.nodes) {
                log.info("{} \u2192 {} prop={} t={}", ev.index(), ev, ev.propensity, ev.time());
                if (Double.isInfinite(ev.time()) && ev.index() + 1 < ((NextEvent[])obj.queue.nodes).length) {
                    log.info("{} \u2014 {} will happen at infinity", ev.index() + 1, ((NextEvent[])obj.queue.nodes).length - 1);
                } else {
                    if (log_start_events < 0 || i++ < log_start_events) continue;
                    log.info("Not showing events {} \u2014 {}", ev.index() + 1, ((NextEvent[])obj.queue.nodes).length - 1);
                }
                break;
            }
        } else {
            log.info("{} event channels", ((NextEvent[])obj.queue.nodes).length);
        }
        NextEventQueue.log_dependency_edges(e);
        if (only_init) {
            System.exit(0);
        }
        return obj;
    }

    public double advance(double time, double tstop, double timelimit, int[][] eventStatistics, List<IGridCalc.Happening> events) {
        double now;
        NextEvent ev = this.queue.first();
        if (ev == null) {
            if (!_warned_empty) {
                log.warn("Event queue is empty \u2014 no diffusion, reaction, or stimulation events");
                _warned_empty = true;
            }
            now = Double.POSITIVE_INFINITY;
        } else {
            now = ev.time;
        }
        assert (now >= time) : ev;
        if (now > tstop) {
            log.debug("Next event is {} time {}, past stop at {}", ev, now, tstop);
            return tstop;
        }
        ev.update(eventStatistics, now, tstop, timelimit, events);
        return now;
    }

    public Collection<IGridCalc.Event> getEvents() {
        return new ArrayList<IGridCalc.Event>(Arrays.asList(this.queue.nodes));
    }

    public class Happening
    implements IGridCalc.Happening {
        final int event_number;
        final IGridCalc.HappeningKind kind;
        final int extent;
        final double time;
        final double waited;
        final double original_wait;

        public Happening(int event_number, IGridCalc.HappeningKind kind, int extent, double time, double waited, double original_wait) {
            this.event_number = event_number;
            this.kind = kind;
            this.extent = extent;
            this.time = time;
            this.waited = waited;
            this.original_wait = original_wait;
        }

        @Override
        public int event_number() {
            return this.event_number;
        }

        @Override
        public IGridCalc.HappeningKind kind() {
            return this.kind;
        }

        @Override
        public int extent() {
            return this.extent;
        }

        @Override
        public double time() {
            return this.time;
        }

        @Override
        public double waited() {
            return this.waited;
        }

        @Override
        public double original_wait() {
            return this.original_wait;
        }
    }

    class IndexDescription {
        public final int position;
        public String description;

        IndexDescription(int position) {
            this.position = position;
            NextEventQueue.this.stat_descriptions.put(position, this);
        }

        public void setDescription(String description) {
            assert (this.description == null || this.description.equals(description)) : this.description + " vs. " + description;
            this.description = description;
        }
    }

    static class IndexOption {
        public final String label;
        public final int ident;

        IndexOption(String label, int ident) {
            this.label = label;
            this.ident = ident;
        }
    }

    public class NextStimulation
    extends NextEvent {
        final int sp;
        final double fraction;
        final StimulationTable.Stimulation stim;

        NextStimulation(int event_number, IndexDescription stat_index, int element, double fraction, int sp, String signature, StimulationTable.Stimulation stim) {
            super(event_number, stat_index, element, signature, new int[0], new int[0]);
            this.sp = sp;
            this.fraction = fraction;
            this.stim = stim;
            this.propensity = this.calcPropensity();
            this.setEvent(1, false, 0.0, this._new_time(0.0));
            log.info("Created {}: t={} fraction={} [{}]", this, this.time, this.fraction, this.stim);
        }

        @Override
        public IGridCalc.EventType event_type() {
            return IGridCalc.EventType.STIMULATION;
        }

        @Override
        int execute(int[][] eventStatistics, int count) {
            NextEventQueue.this.updatePopulation(this.element(), this.sp, count, this);
            this.updateStatistics(eventStatistics, count);
            return count;
        }

        private double _continous_delta_to_real_time(double current, double delta, boolean insideDuration) {
            double t2;
            double tp;
            double tc;
            if (Double.isNaN(this.stim.period)) {
                tc = this.stim.onset;
                tp = Math.max(current - this.stim.onset, 0.0);
            } else {
                double nc = (current - this.stim.onset) / this.stim.period;
                if (nc < 0.0) {
                    nc = 0.0;
                }
                tp = nc % 1.0 * this.stim.period;
                assert (current > this.stim.onset || tp == 0.0);
                if (tp < this.stim.duration) {
                    tc = this.stim.onset + Math.floor(nc) * this.stim.period;
                } else {
                    tc = this.stim.onset + Math.ceil(nc) * this.stim.period;
                    tp = 0.0;
                }
            }
            double t1 = tp + delta;
            if (insideDuration && t1 > this.stim.duration) {
                return tc + this.stim.duration;
            }
            if (Double.isNaN(this.stim.period)) {
                t1 += tc;
            } else {
                int n = (int)(t1 / this.stim.duration);
                t1 = tc + (double)n * this.stim.period + t1 % this.stim.duration;
            }
            double d = t2 = t1 < this.stim.end ? t1 : Double.POSITIVE_INFINITY;
            assert (insideDuration || t2 + 1.0E-6 >= current + delta) : "t1=" + t1 + " t2=" + t2 + " current=" + current + " delta=" + delta + " current+delta=" + (current + delta);
            if (!insideDuration && t2 < current) {
                return current;
            }
            return t2;
        }

        @Override
        double exact_time(double current) {
            double cont = super.exact_time(current);
            double real = this._continous_delta_to_real_time(current, cont, false);
            return real - current;
        }

        @Override
        double _new_time(double current) {
            double time;
            switch (this.stim.distribution) {
                case POISSON: {
                    time = super._new_time(0.0);
                    break;
                }
                case EXACT: {
                    time = 1.0 / this.propensity;
                    break;
                }
                default: {
                    throw new RuntimeException("not implemented");
                }
            }
            return this._continous_delta_to_real_time(current, time, false);
        }

        @Override
        public double calcPropensity() {
            assert (this.sp == this.stim.species);
            double ans = this.stim.rate * this.fraction;
            assert (ans >= 0.0) : ans;
            return ans;
        }

        @Override
        public int[] substrates() {
            return new int[]{this.sp};
        }

        @Override
        public int[] substrate_stoichiometry() {
            return PLUS_ONE;
        }

        @Override
        public Map<Integer, int[][]> substrates_by_voxel() {
            HashMap<Integer, int[][]> map = new HashMap<Integer, int[][]>();
            map.put(this.element(), new int[][]{this.substrates(), this.substrate_stoichiometry()});
            return map;
        }

        @Override
        public int element2() {
            return this.element();
        }

        @Override
        public double _update_propensity(boolean warn) {
            return this.propensity;
        }

        @Override
        public double leap_time(double current) {
            double limit1 = this.size1_leap_extent();
            if (limit1 < 1.0 / NextEventQueue.this.tolerance) {
                log.debug("leap time: maximum size1 extent {}, not leaping", limit1);
                return 0.0;
            }
            int limit2 = NextEventQueue.this.particles[this.element()][this.sp];
            double cont_leap_time = NextEventQueue.this.tolerance * Math.min(limit1, (double)limit2) / this.propensity;
            assert (!(cont_leap_time < 0.0));
            double until = this._continous_delta_to_real_time(current, cont_leap_time, true);
            log.debug("{}: leap time: {}\u00d7min({}, {})/{} \u2192 {} cont, {} real until {}", this, NextEventQueue.this.tolerance, limit1, limit2, this.propensity, cont_leap_time, until - current, until);
            if (until <= current) {
                return 0.0;
            }
            return until - current;
        }

        @Override
        public int leap_count(double current, double time, boolean bidirectional) {
            assert (!bidirectional);
            switch (this.stim.distribution) {
                case POISSON: {
                    return NextEventQueue.this.stepper.poisson(this.propensity * time);
                }
                case EXACT: {
                    return NextEventQueue.this.random.round(this.propensity * time);
                }
            }
            throw new RuntimeException("not implemented");
        }

        @Override
        public void addRelations(HashMap<Integer, ArrayList<NextEvent>> map, String[] species, boolean verbose) {
            for (NextEvent e : map.get(this.element())) {
                assert (e.element() == this.element());
                if (e == this || !ArrayUtil.intersect(e.reactants(), this.sp)) continue;
                this.addDependent(e, species, verbose);
            }
        }

        public String toString() {
            return String.format("Stimulation el.%d %s", this.element(), this.signature);
        }
    }

    public class NextReaction
    extends NextEvent {
        final int[] products;
        final int[] product_stoichiometry;
        final int[] reactant_powers;
        final int[] substrates;
        final int[] substrate_stoichiometry;
        final int index;
        final double rate;
        final double volume;

        NextReaction(int event_number, IndexDescription stat_index, int index, int element, int[] reactants, int[] products, int[] reactant_stoichiometry, int[] product_stoichiometry, int[] reactant_powers, String signature, double rate, double volume) {
            super(event_number, stat_index, element, signature, reactants, reactant_stoichiometry);
            this.index = index;
            this.products = products;
            this.product_stoichiometry = product_stoichiometry;
            this.reactant_powers = reactant_powers;
            this.rate = rate;
            this.volume = volume;
            int[][] tmp = NextEventQueue.stoichiometry(reactants, reactant_stoichiometry, products, product_stoichiometry);
            this.substrates = tmp[0];
            this.substrate_stoichiometry = tmp[1];
            this.propensity = this.calcPropensity();
            this.setEvent(1, false, 0.0, this.propensity > 0.0 ? this._new_time(0.0) : Double.POSITIVE_INFINITY);
            log.debug("Created {} rate={} vol={} time={}", this, this.rate, this.volume, this.time);
            assert (this.time >= 0.0);
        }

        @Override
        public IGridCalc.EventType event_type() {
            return IGridCalc.EventType.REACTION;
        }

        protected int[] productPopulation() {
            return ArrayUtil.pick(NextEventQueue.this.particles[this.element()], this.products);
        }

        protected double self_leap_limit(int[] X) {
            int[] reactants = this.reactants();
            if (reactants.length == 0) {
                return Double.POSITIVE_INFINITY;
            }
            if (reactants.length == 1) {
                return (double)X[reactants[0]] / (double)(this.reactant_powers[0] * this.reactant_stoichiometry()[0]);
            }
            double mult = 0.0;
            for (int i = 0; i < reactants.length; ++i) {
                mult += (double)this.reactant_powers[i] * (double)this.reactant_stoichiometry()[0] / (double)X[reactants[i]];
            }
            return 1.0 / mult;
        }

        @Override
        public double leap_time(double current) {
            double effective_propensity;
            double limit1 = this.size1_leap_extent();
            if (limit1 < 1.0 / NextEventQueue.this.tolerance) {
                log.debug("leap time: maximum size1 extent {}, not leaping", limit1);
                return 0.0;
            }
            int[] X = NextEventQueue.this.particles[this.element()];
            double limit2 = this.self_leap_limit(X);
            double time = limit2 / this.propensity;
            double limit3 = -1.0;
            if (this.reverse == null) {
                effective_propensity = this.propensity;
            } else {
                limit3 = ((NextReaction)this.reverse).self_leap_limit(X);
                time = Math.min(time, limit3 / this.reverse.propensity);
                effective_propensity = Math.abs(this.propensity - this.reverse.propensity);
            }
            time = NextEventQueue.this.tolerance * Math.min(limit1 / effective_propensity, time);
            log.debug("{}: leap time: subs {}\u00d7{}, \u025b={}, pop.{}\u2192{} \u2192 limit {},{},{} \u2192 leap={}", this, this.substrates, this.substrate_stoichiometry, NextEventQueue.this.tolerance, this.reactantPopulation(), this.productPopulation(), limit1, limit2, limit3 == -1.0 ? "-" : Double.valueOf(limit3), time);
            if (this.stimulations != null) {
                NextStimulation first = null;
                for (NextStimulation stim : this.stimulations) {
                    if (!(stim.time < current + time) || first != null && !(stim.time < first.time)) continue;
                    first = stim;
                }
                if (first != null) {
                    double oldans = time;
                    time = first.time - current + 1.0E-12;
                    log.debug("leap time: curtailing {} by next {} to {} (from {})", this, first, time, oldans);
                }
            }
            assert (!(time < 0.0)) : time;
            return time;
        }

        private int leap_count_uni(int[] X, double time) {
            int n = Integer.MAX_VALUE;
            for (int i = 0; i < this.reactants().length; ++i) {
                n = Math.min(n, X[this.reactants()[i]] / this.reactant_stoichiometry()[i]);
            }
            return NextEventQueue.this.stepper.versatile_ngo(n, this.propensity * time / (double)n);
        }

        @Override
        public int leap_count(double current, double time, boolean bidirectional) {
            int[] X = NextEventQueue.this.particles[this.element()];
            int n1 = this.leap_count_uni(X, time);
            if (!bidirectional) {
                return n1;
            }
            assert (this.reverse != null);
            assert (this.element() == this.reverse.element());
            int n2 = ((NextReaction)this.reverse).leap_count_uni(X, time);
            return n1 - n2;
        }

        private void maybeAddRelation(NextEvent e, String[] species, boolean verbose) {
            for (int r1 : e.reactants()) {
                for (int i = 0; i < this.substrates.length; ++i) {
                    if (this.substrates[i] != r1) continue;
                    this.addDependent(e, species, verbose);
                    return;
                }
            }
        }

        @Override
        public void addRelations(HashMap<Integer, ArrayList<NextEvent>> map, String[] species, boolean verbose) {
            for (NextEvent e : map.get(this.element())) {
                assert (e.element() == this.element());
                if (e == this) continue;
                this.maybeAddRelation(e, species, verbose);
                if (!(e instanceof NextStimulation)) continue;
                this.addStimulation((NextStimulation)e);
            }
        }

        @Override
        int execute(int[][] eventStatistics, int count) {
            int i;
            for (i = 0; i < this.reactants().length; ++i) {
                if (NextEventQueue.this.particles[this.element()][this.reactants()[i]] >= this.reactant_stoichiometry()[i] * count) continue;
                int oldcount = count;
                count = NextEventQueue.this.particles[this.element()][this.reactants()[i]] / this.reactant_stoichiometry()[i];
                log.warn("{}: population would go below zero with prop={} reactants {}\u00d7{} extent={} (using {})", this, this.propensity, this.reactantPopulation(), this.reactant_powers, oldcount, count);
            }
            for (i = 0; i < this.reactants().length; ++i) {
                NextEventQueue.this.updatePopulation(this.element(), this.reactants()[i], this.reactant_stoichiometry()[i] * -count, this);
            }
            for (i = 0; i < this.products.length; ++i) {
                NextEventQueue.this.updatePopulation(this.element(), this.products[i], this.product_stoichiometry[i] * count, this);
            }
            this.updateStatistics(eventStatistics, count);
            return count;
        }

        @Override
        public double calcPropensity() {
            double ans = AdaptiveGridCalc.calculatePropensity(this.reactants(), this.products, this.reactant_stoichiometry(), this.product_stoichiometry, this.reactant_powers, this.rate, this.volume, NextEventQueue.this.particles[this.element()]);
            assert (ans >= 0.0) : ans;
            return ans;
        }

        @Override
        public int[] substrates() {
            assert (this.substrates != null);
            return this.substrates;
        }

        @Override
        public int[] substrate_stoichiometry() {
            assert (this.substrate_stoichiometry != null);
            return this.substrate_stoichiometry;
        }

        @Override
        public Map<Integer, int[][]> substrates_by_voxel() {
            HashMap<Integer, int[][]> map = new HashMap<Integer, int[][]>();
            map.put(this.element(), new int[][]{this.substrates(), this.substrate_stoichiometry()});
            return map;
        }

        @Override
        public int element2() {
            return this.element();
        }

        public String toString() {
            return String.format("Reaction el.%d %s", this.element(), this.signature);
        }
    }

    public class NextDiffusion
    extends NextEvent {
        final int element2;
        final int index2;
        final int sp;
        final double fdiff;

        NextDiffusion(int event_number, IndexDescription stat_index, int element, int element2, int index2, int sp, String signature, double fdiff) {
            super(event_number, stat_index, element, signature, new int[]{sp}, new int[]{1});
            this.element2 = element2;
            this.index2 = index2;
            this.sp = sp;
            this.fdiff = fdiff;
            this.propensity = this.calcPropensity();
            this.setEvent(1, false, 0.0, this.propensity > 0.0 ? this._new_time(0.0) : Double.POSITIVE_INFINITY);
            log.debug("Created {}: t={}", this, this.time);
        }

        @Override
        public IGridCalc.EventType event_type() {
            return IGridCalc.EventType.DIFFUSION;
        }

        @Override
        int execute(int[][] eventStatistics, int count) {
            int done = NextEventQueue.this.updatePopulation(this.element(), this.sp, -count, this);
            NextEventQueue.this.updatePopulation(this.element2, this.sp, -done, this);
            this.updateStatistics(eventStatistics, done);
            return -done;
        }

        @Override
        public double calcPropensity() {
            double ans = this.fdiff * (double)NextEventQueue.this.particles[this.element()][this.sp];
            assert (ans >= 0.0) : ans;
            return ans;
        }

        @Override
        public int[] substrates() {
            return new int[]{this.sp, this.sp};
        }

        @Override
        public int[] substrate_stoichiometry() {
            return MINUS_ONE_PLUS_ONE;
        }

        @Override
        public int element2() {
            return this.element2;
        }

        @Override
        public Map<Integer, int[][]> substrates_by_voxel() {
            HashMap<Integer, int[][]> map = new HashMap<Integer, int[][]>();
            map.put(this.element(), new int[][]{this.reactants(), MINUS_ONE});
            map.put(this.element2, new int[][]{this.reactants(), PLUS_ONE});
            return map;
        }

        @Override
        public double leap_time(double current) {
            double ans;
            double limit1 = this.size1_leap_extent();
            if (limit1 < 1.0 / NextEventQueue.this.tolerance) {
                log.debug("leap time: maximum size1 extent {}, not leaping", limit1);
                return 0.0;
            }
            int X1 = NextEventQueue.this.particles[this.element()][this.sp];
            int X2 = NextEventQueue.this.particles[this.element2][this.sp];
            int Xm = Math.min(X1, X2);
            int Xtotal = X1 + X2;
            double limit = NextEventQueue.this.tolerance * Math.min(limit1, (double)Xm);
            double r1 = this.fdiff;
            double r2 = ((NextDiffusion)this.reverse).fdiff;
            double t1 = limit / Math.abs(r1 * (double)X1 - r2 * (double)X2);
            log.debug("diff propensity: {}, {}, r1={}, r2={} \u2192 propensity={}-{}={}", X1, X2, r1, r2, this.propensity, this.reverse.propensity, this.propensity - this.reverse.propensity);
            double arg = 1.0 - limit * limit * (r1 + r2) / (r1 * (double)X1 + r2 * (double)X2);
            if (arg > 0.0) {
                double t2 = Math.log(arg) / -(r1 + r2);
                ans = Math.min(t1, t2);
                log.debug("leap time: min({}, {}, limit {}, {}: E\u2192{}, V\u2192{}) \u2192 {}", X1, X2, limit1, Xm, t1, t2, ans);
            } else {
                ans = t1;
                log.debug("leap time: min({}, {}, limit {}, {}: E\u2192{}, V\u2192inf) \u2192 {}", X1, X2, limit1, Xm, t1, ans);
            }
            if (this.stimulations != null) {
                NextStimulation first = null;
                for (NextStimulation stim : this.stimulations) {
                    if (!(stim.time < current + ans) || first != null && !(stim.time < first.time)) continue;
                    first = stim;
                }
                if (first != null) {
                    double oldans = ans;
                    ans = first.time - current + 1.0E-12;
                    log.debug("leap time: curtailing {} by next {} to {} (from {})", this, first, ans, oldans);
                }
            }
            return ans;
        }

        @Override
        public int leap_count(double current, double time, boolean bidirectional) {
            int X1 = NextEventQueue.this.particles[this.element()][this.sp];
            double r1 = this.fdiff;
            if (!bidirectional) {
                return NextEventQueue.this.stepper.versatile_ngo(X1, time * r1);
            }
            double r2 = ((NextDiffusion)this.reverse).fdiff;
            double r12 = r1 + r2;
            int X2 = NextEventQueue.this.particles[this.element2][this.sp];
            double mult = -Math.expm1(-r12 * time) / r12;
            int n1 = NextEventQueue.this.stepper.versatile_ngo(X1, r1 * mult);
            int n2 = NextEventQueue.this.stepper.versatile_ngo(X2, r2 * mult);
            return n1 - n2;
        }

        @Override
        public void addRelations(HashMap<Integer, ArrayList<NextEvent>> map, String[] species, boolean verbose) {
            HashSet set = new HashSet(map.get(this.element()));
            set.addAll(map.get(this.element2));
            for (NextEvent e : set) {
                if (e == this) continue;
                assert (e.element() == this.element() || e.element() == this.element2);
                if (ArrayUtil.intersect(e.reactants(), this.sp)) {
                    this.addDependent(e, species, verbose);
                    continue;
                }
                if (!(e instanceof NextStimulation)) continue;
                this.addStimulation((NextStimulation)e);
            }
        }

        public String toString() {
            return String.format("Diffusion %s el.%d\u2192%d", this.signature, this.element(), this.element2);
        }
    }

    public abstract class NextEvent
    implements Node,
    IGridCalc.Event {
        int index;
        private final int event_number;
        final IndexDescription stat_index;
        private final int element;
        final String signature;
        private final int[] reactants;
        private final int[] reactant_stoichiometry;
        protected List<ScoeffElem> scoeff_ki = new ArrayList<ScoeffElem>();
        private double wait_start;
        protected double time;
        protected double original_wait;
        protected int extent;
        protected boolean leap;
        protected NextEvent reverse;
        protected boolean reverse_is_leaping;
        double propensity;
        Happening happening;
        int[] old_pop;
        private boolean _log_level_enabled = false;
        List<NextEvent> dependent = new ArrayList<NextEvent>();
        List<NextEvent> dependon = new ArrayList<NextEvent>();
        List<NextStimulation> stimulations = null;

        @Override
        public abstract IGridCalc.EventType event_type();

        NextEvent(int event_number, IndexDescription stat_index, int element, String signature, int[] reactants, int[] reactant_stoichiometry) {
            for (int i : reactant_stoichiometry) {
                assert (i > 0);
            }
            this.event_number = event_number;
            this.stat_index = stat_index;
            this.element = element;
            this.signature = signature;
            this.reactants = reactants;
            this.reactant_stoichiometry = reactant_stoichiometry;
        }

        protected void setEvent(int extent, boolean leap, double wait_start, double time) {
            assert (!this.reverse_is_leaping);
            this.extent = extent;
            this.leap = leap;
            this.wait_start = wait_start;
            this.time = time;
            this.original_wait = time - wait_start;
            assert (time >= 0.0) : this;
        }

        @Override
        public int event_number() {
            return this.event_number;
        }

        @Override
        public int stat_index() {
            if (this.stat_index != null) {
                return this.stat_index.position;
            }
            return -1;
        }

        @Override
        public String stat_index_description() {
            if (this.stat_index != null) {
                return this.stat_index.description;
            }
            return null;
        }

        @Override
        public int index() {
            return this.index;
        }

        @Override
        public int element() {
            return this.element;
        }

        @Override
        public String description() {
            return this.toString();
        }

        @Override
        public void setIndex(int index) {
            this.index = index;
        }

        @Override
        public double time() {
            return this.time;
        }

        @Override
        public abstract int[] substrates();

        @Override
        public abstract int[] substrate_stoichiometry();

        public abstract Map<Integer, int[][]> substrates_by_voxel();

        public void addReverse(NextEvent other) {
            if (this.reverse != null) {
                throw new RuntimeException(String.format("%s trying to add %s as reverse but %s was added before", this, other, this.reverse));
            }
            if (other.reverse != null) {
                throw new RuntimeException(String.format("%s added to %s as reverse but %s was added before", this, other, other.reverse));
            }
            this.reverse = other;
            other.reverse = this;
        }

        abstract int execute(int[][] var1, int var2);

        abstract double calcPropensity();

        abstract double leap_time(double var1);

        double size1_leap_extent() {
            int[] subs = this.substrates();
            double min_value = Double.POSITIVE_INFINITY;
            for (ScoeffElem scoeff : this.scoeff_ki) {
                if (scoeff.single_coeff > 0) {
                    double val = (double)NextEventQueue.this.particles[scoeff.element][scoeff.single_sub] / (double)scoeff.single_coeff;
                    min_value = Math.min(min_value, val);
                    continue;
                }
                double change = 0.0;
                for (int n = 0; n < subs.length; ++n) {
                    change += (double)scoeff.coeff[n] / (double)NextEventQueue.this.particles[scoeff.element][subs[n]];
                }
                change = Math.abs(change);
                min_value = Math.min(min_value, 1.0 / change);
            }
            return min_value;
        }

        double exact_time(double current) {
            return 1.0 / this.propensity;
        }

        abstract int leap_count(double var1, double var3, boolean var5);

        double _new_time(double current) {
            double exp = NextEventQueue.this.random.exponential(this.propensity);
            if (this.propensity > 0.0) {
                log.debug("exponential time for prop={} \u2192 time={}", this.propensity, exp);
            }
            return current + exp;
        }

        double _update_propensity(boolean warn) {
            double old = this.propensity;
            this.propensity = this.calcPropensity();
            if (check_updates) {
                int[] pop = this.reactantPopulation();
                if (this.old_pop != null) {
                    if (warn && this.propensity != 0.0 && this.propensity == old) {
                        boolean higher = false;
                        boolean lower = false;
                        for (int i = 0; i < pop.length; ++i) {
                            if (pop[i] < this.old_pop[i]) {
                                lower = true;
                            }
                            if (pop[i] <= this.old_pop[i]) continue;
                            higher = true;
                        }
                        log.log(higher && lower ? Level.DEBUG : Level.ERROR, "{}: propensity changed {} \u2192 {} ({}, n={} \u2192 {}), extent={}", this, old, this.propensity, (this.propensity - old) / old, this.old_pop, pop, this.extent);
                        if (!higher || !lower) {
                            throw new RuntimeException();
                        }
                    } else if (log_propensity) {
                        log.debug("{}: propensity changed {} \u2192 {} ({}, n={} \u2192 {}), extent={}", this, old, this.propensity, (this.propensity - old) / old, this.old_pop, pop, this.extent);
                    }
                }
                this.old_pop = pop;
            }
            return old;
        }

        protected int[] reactantPopulation() {
            return ArrayUtil.pick(NextEventQueue.this.particles[this.element()], this.reactants());
        }

        void pick_time(double current, double timelimit) {
            double exact = this.exact_time(current);
            if (NextEventQueue.this.adaptive) {
                double leap = this.leap_time(current);
                log.debug("options: eff.prop={}, wait {}, leap {}", this.propensity - (this.reverse != null ? this.reverse.propensity : 0.0), exact, leap);
                if (current + leap > timelimit) {
                    log.debug("Curtailing leap {}\u2192{} to {}", current, current + leap, timelimit);
                    leap = timelimit - current;
                }
                if (leap > exact * NextEventQueue.this.leap_min_jump) {
                    double prob;
                    assert (update_times);
                    boolean bidirectional = this.reverse != null;
                    int count = this.leap_count(current, leap, bidirectional);
                    if (bidirectional && this.reverse.time >= current && this.reverse.time < current + leap && (prob = 1.0 - Math.exp(-this.reverse.propensity * (current - this.reverse.wait_start)) * (1.0 - Math.exp(-this.reverse.propensity * leap))) > (double)NextEventQueue.this.random.random()) {
                        --count;
                    }
                    log.debug("{}: leaping {} {} ({}\u2192{}), extent {}", this, bidirectional ? "bi" : "uni", leap, current, current + leap, count);
                    this.setEvent(count, true, current, current + leap);
                    return;
                }
            }
            double normal = this._new_time(current);
            assert (!Double.isNaN(current));
            assert (!Double.isNaN(normal));
            log.debug("waiting {} {}\u2192{}", normal - current, current, normal);
            this.setEvent(1, false, current, normal);
        }

        void update_dependent(boolean reverse, double current, int leap_extent) {
            double max_fraction = 0.0;
            NextEvent worst = null;
            for (NextEvent dep : this.dependent) {
                double fraction;
                if (!reverse && dep == this.reverse || !((fraction = dep.update_and_reposition(current, true)) > max_fraction)) continue;
                max_fraction = fraction;
                worst = dep;
            }
            if (leap_extent != 0 && max_fraction >= 5.0 * NextEventQueue.this.tolerance) {
                double propensity = this.propensity - (this.reverse != null && this.leap ? this.reverse.propensity : 0.0);
                double linear = propensity * this.original_wait;
                double ratio = max_fraction / Math.abs((double)leap_extent / linear);
                Level level = ratio < 2.0 ? Level.INFO : Logging.NOTICE;
                log.log(level, "max change fraction {} @ {}:\n        {}, extent {} (\u00b5={}, pop={})\n        for {} (pop={})\n        reverse {} (pop={})", max_fraction, current, this, leap_extent, linear, this.reactantPopulation(), worst, worst.reactantPopulation(), worst.reverse != null ? worst.reverse : "(none)", worst.reverse != null ? (Object)worst.reverse.reactantPopulation() : "");
            }
        }

        void update(int[][] eventStatistics, double current, double tstop, double timelimit, List<IGridCalc.Happening> events) {
            int done;
            assert (this.reverse != null || this.extent >= 0) : this.extent;
            boolean was_leap = this.leap;
            if (!this._log_level_enabled && current >= log_debug_start) {
                Logging.setLogLevel(null, "neurord.numeric.grid.NextEventQueue", Level.DEBUG);
                this._log_level_enabled = true;
            }
            if (events != null) {
                events.add(new Happening(this.event_number, this.leap ? IGridCalc.HappeningKind.LEAP : IGridCalc.HappeningKind.EXACT, this.extent, current, current - this.wait_start, this.original_wait));
            }
            if (!this.leap) {
                done = this.execute(eventStatistics, this.extent);
                if (done != 0) {
                    this._update_propensity(false);
                    this.update_dependent(true, current, 0);
                }
            } else {
                done = this.extent;
            }
            if (was_leap) {
                ++NextEventQueue.this.leaps;
                NextEventQueue.this.leap_extent += (long)Math.abs(done);
            } else {
                ++NextEventQueue.this.normal_waits;
            }
            log.debug("Advanced to {} with {} {}extent={}{}", this.time, this, was_leap ? "leap " : "", done, done == this.extent ? "" : " (planned " + this.extent + ")");
            this.pick_time(current, timelimit);
            NextEventQueue.this.queue.reposition("update", this);
            if (was_leap != this.leap && this.reverse != null) {
                assert (this.reverse.reverse_is_leaping == was_leap);
                if (this.leap) {
                    this.reverse.setEvent(1, false, current, Double.POSITIVE_INFINITY);
                    this.reverse.reverse_is_leaping = true;
                    NextEventQueue.this.queue.reposition("reverse", this.reverse);
                } else {
                    this.reverse.reverse_is_leaping = false;
                    this.reverse.update_and_reposition(current, false);
                }
            }
            if (this.leap && (done = this.execute(eventStatistics, this.extent)) != 0) {
                this._update_propensity(false);
                if (this.reverse != null) {
                    this.reverse._update_propensity(false);
                }
                this.update_dependent(false, current, done);
            }
        }

        double update_and_reposition(double current, boolean changed) {
            double ans;
            boolean inf;
            if (this.reverse_is_leaping) {
                if (log_reposition) {
                    log.debug("update_and_reposition: {}, doing reverse", this);
                }
                assert (!this.reverse.reverse_is_leaping) : this.reverse;
                return this.reverse.update_and_reposition(current, false);
            }
            if (log_reposition) {
                log.debug("update_and_reposition: {}", this);
            }
            boolean expect_changed = changed && (this.reverse == null || !this.leap);
            double old = this._update_propensity(expect_changed);
            boolean bl = inf = Double.isInfinite(this.time) || this.propensity == 0.0;
            if (update_times && !inf) {
                this.time = (this.time - current) * old / this.propensity + current;
                ans = (this.propensity - old) / old;
            } else {
                this.time = this._new_time(current);
                if (inf) {
                    this.original_wait = this.time - current;
                }
                ans = 0.0;
            }
            assert (this.time >= 0.0) : this;
            NextEventQueue.this.queue.reposition("upd.dep", this);
            return ans;
        }

        @Override
        public Collection<IGridCalc.Event> dependent() {
            return new ArrayList<IGridCalc.Event>(this.dependent);
        }

        public int[] reactants() {
            return this.reactants;
        }

        public int[] reactant_stoichiometry() {
            return this.reactant_stoichiometry;
        }

        protected void addDependent(NextEvent ev, String[] species, boolean verbose) {
            int[] stoichio;
            int[] subs;
            int elem;
            assert (!this.dependent.contains(ev)) : this;
            this.dependent.add(ev);
            ev.dependon.add(this);
            if (ev == this.reverse) {
                if (verbose) {
                    log.debug("      \u2192 {}:{} [reverse el.{}]", ev.index(), ev, ev.element());
                }
                return;
            }
            for (Map.Entry<Integer, int[][]> entry : this.substrates_by_voxel().entrySet()) {
                elem = entry.getKey();
                subs = entry.getValue()[0];
                stoichio = entry.getValue()[1];
                if (elem != ev.element()) continue;
                ScoeffElem scoeff = ScoeffElem.create(elem, subs, stoichio, ev.reactants(), ev.reactant_stoichiometry());
                boolean same = false;
                for (ScoeffElem other : this.scoeff_ki) {
                    if (!other.sameAs(scoeff)) continue;
                    same = true;
                    break;
                }
                if (!same) {
                    this.scoeff_ki.add(scoeff);
                }
                if (verbose) {
                    int sum = ArrayUtil.abssum(scoeff.coeff);
                    String formula = scoeff.toString(subs, species);
                    log.debug("      \u2192 {}:{} [{}{}]", ev.index(), ev, formula, same ? " duplicate" : "");
                }
                return;
            }
            for (Map.Entry<Integer, int[][]> entry : this.substrates_by_voxel().entrySet()) {
                elem = entry.getKey();
                subs = entry.getValue()[0];
                stoichio = entry.getValue()[1];
                log.info("substrates_by_voxel: el.{} \u2192 {} \u00d7 {}", elem, stoichio, subs);
            }
            log.error("Dependency not found: {} dep. {}", this, ev);
            throw new RuntimeException("wtf?");
        }

        void addStimulation(NextStimulation ev) {
            if (this.stimulations == null) {
                this.stimulations = new ArrayList<NextStimulation>();
            }
            this.stimulations.add(ev);
            log.debug("{}: adding dependency on stimulation {}", this, ev);
        }

        public abstract void addRelations(HashMap<Integer, ArrayList<NextEvent>> var1, String[] var2, boolean var3);

        protected void updateStatistics(int[][] eventStatistics, int firings) {
            if (eventStatistics == null) {
                return;
            }
            if (this.stat_index == null) {
                return;
            }
            int[] nArray = eventStatistics[this.stat_index.position];
            nArray[0] = nArray[0] + 1;
            int[] nArray2 = eventStatistics[this.stat_index.position];
            nArray2[1] = nArray2[1] + firings;
        }
    }

    static class ScoeffElem {
        final int element;
        final int single_coeff;
        final int single_sub;
        final int[] coeff;

        ScoeffElem(int element, int[] coeff, int[] substrates) {
            this.element = element;
            this.coeff = coeff;
            int single_coeff = 0;
            int single_sub = -1;
            for (int i = 0; i < coeff.length; ++i) {
                if (coeff[i] == 0) continue;
                if (single_coeff == 0) {
                    single_coeff = Math.abs(coeff[i]);
                    single_sub = substrates[i];
                    continue;
                }
                single_coeff = 0;
                single_sub = -1;
                break;
            }
            this.single_coeff = single_coeff;
            this.single_sub = single_sub;
        }

        public boolean sameAs(ScoeffElem other) {
            return this.element == other.element && (Arrays.equals(this.coeff, other.coeff) || Arrays.equals(this.coeff, ArrayUtil.negate(other.coeff)));
        }

        public String toString(int[] subs, String[] species) {
            assert (subs.length == this.coeff.length);
            StringBuilder b = new StringBuilder();
            for (int i = 0; i < subs.length; ++i) {
                if (this.coeff[i] == 0) continue;
                if (b.length() > 0) {
                    b.append(" + ");
                }
                b.append("" + this.coeff[i] + "/" + species[subs[i]]);
            }
            if (this.single_coeff > 0) {
                b.append(" \ud83d\ude0a");
            }
            b.append(" el." + this.element);
            return b.toString();
        }

        public static int[] scoeff_ki(int[] substrates, int[] substrate_stoichiometry, int[] reactants, int[] reactant_stoichiometry) {
            assert (substrates.length == substrate_stoichiometry.length) : Arrays.toString(substrates) + " vs. " + Arrays.toString(substrate_stoichiometry);
            assert (reactants.length == reactant_stoichiometry.length) : Arrays.toString(reactants) + " vs. " + Arrays.toString(reactant_stoichiometry);
            int[] ans = new int[substrates.length];
            block0: for (int i = 0; i < substrates.length; ++i) {
                for (int ii = 0; ii < reactants.length; ++ii) {
                    if (reactants[ii] != substrates[i]) continue;
                    ans[i] = substrate_stoichiometry[i] * reactant_stoichiometry[ii];
                    continue block0;
                }
            }
            return ans;
        }

        public static ScoeffElem create(int element, int[] substrates, int[] substrate_stoichiometry, int[] reactants, int[] reactant_stoichiometry) {
            return new ScoeffElem(element, ScoeffElem.scoeff_ki(substrates, substrate_stoichiometry, reactants, reactant_stoichiometry), substrates);
        }
    }

    public class PriorityTree<T extends Node> {
        T[] nodes;
        long swaps = 0L;

        protected T child(T a, int which) {
            assert (which < 2);
            int ch = (a.index() + 1) * 2 - 1 + which;
            return ch < this.nodes.length ? (T)this.nodes[ch] : null;
        }

        protected T parent(T a) {
            int ch = (a.index() + 1) / 2 - 1;
            if (ch < 0) {
                return null;
            }
            return this.nodes[ch];
        }

        protected T littlestChild(T a) {
            T child = this.child(a, 0);
            if (child == null) {
                return null;
            }
            T child2 = this.child(a, 1);
            if (child2 == null) {
                return child;
            }
            if (child.time() <= child2.time()) {
                return child;
            }
            return child2;
        }

        void swap(T a, T b) {
            assert (this.parent(b) == a);
            int ai = a.index();
            int bi = b.index();
            this.nodes[ai] = b;
            this.nodes[bi] = a;
            a.setIndex(bi);
            b.setIndex(ai);
            ++this.swaps;
        }

        void build(T[] nodes) {
            if (!update_times) {
                log.info("neurord.neq.update_times is false, will regenerate times");
            }
            Comparator c = new Comparator<T>(){

                @Override
                public int compare(T a, T b) {
                    return Double.compare(a.time(), b.time());
                }
            };
            log.debug("Sorting {} nodes", nodes.length);
            Arrays.sort(nodes, c);
            for (int i = 0; i < nodes.length; ++i) {
                nodes[i].setIndex(i);
            }
            this.nodes = nodes;
        }

        T first() {
            if (this.nodes.length == 0) {
                return null;
            }
            T node = this.nodes[0];
            assert (node != null);
            return node;
        }

        void reposition(String prefix, T node) {
            assert (node != null);
            T parent = this.parent(node);
            if (log_queue) {
                log.debug("{}: moving {} t={} parent={}", prefix, node, node.time(), parent);
            }
            if (parent != null && parent.time() > node.time()) {
                this.swap(parent, node);
                this.reposition(prefix, node);
            } else {
                T littlest = this.littlestChild(node);
                if (log_queue) {
                    log.debug("littlest child is {} t={}", littlest, littlest != null ? Double.valueOf(littlest.time()) : "-");
                }
                if (littlest != null && node.time() > littlest.time()) {
                    this.swap(node, littlest);
                    this.reposition(prefix + "-", node);
                }
            }
        }
    }

    public static interface Node {
        public int index();

        public void setIndex(int var1);

        public double time();
    }

    public static class Numbering {
        int count = 0;

        public int get() {
            return this.count++;
        }
    }
}

