TITLE Slow Ca-dependent cation current
:
:   Ca++ dependent nonspecific cation current ICAN
:   Differential equations
:
:   Model based on a first order kinetic scheme
:
:      <closed> + n cai <-> <open>	(alpha,beta)
:
:   Following this model, the activation fct will be half-activated at 
:   a concentration of Cai = (beta/alpha)^(1/n) = cac (parameter)
:
:   The mod file is here written for the case n=2 (2 binding sites)
:   ---------------------------------------------
:
:   Kinetics based on: Partridge & Swandulla, TINS 11: 69-72, 1988.
:
:   This current has the following properties:
:      - inward current (non specific for cations Na, K, Ca, ...)
:      - activated by intracellular calcium
:      - NOT voltage dependent
:
:   A minimal value for the time constant has been added
:
:   Ref: Destexhe et al., J. Neurophysiology 72: 803-818, 1994.
:
:   Modifications by Arthur Houweling for use in MyFirstNEURON
:
:   Some parameter changes by Paulo Aguiar (pauloaguiar@fc.up.pt):
:   tau_factor = 40 => parameter beta	changes from 2.0e-3 to 5.0e-5
:		cac = 0.5e-3 (before was 1.0e-3)

INDEPENDENT {t FROM 0 TO 1 WITH 1 (ms)}

NEURON {
	SUFFIX iCaAN
	USEION can READ ecan WRITE ican VALENCE 1
	USEION ca READ cai
        RANGE gbar, m_inf, tau_m
	RANGE ican
	GLOBAL beta, cac, taumin
}


UNITS {
	(mA) = (milliamp)
	(mV) = (millivolt)
	(molar) = (1/liter)
	(mM) = (millimolar)
}


PARAMETER {
	v		  (mV)
	celsius		  (degC)
        dt                (ms)
	ecan	= -20	  (mV)		: reversal potential
	cai		  (mM)
	gbar	= 0.00025 (mho/cm2)
	beta	= 2.0e-3  (1/ms)	: backward rate constant (original value)

	tau_factor = 40         : scaling factor allowing tuning

	:cac	= 1.0e-3  (mM)		: middle point of activation fct  (original value)
	cac		= 5e-4	  (mM)		: middle point of activation fct

	taumin	= 0.1	  (ms)		: minimal value of time constant
	
	:parameter tau_factor and cac were set to produce tau_m ~ 2000(ms) at cai=cac and celsius=36;
	:implications of parameters change when cai=cac:
	: -> BEFORE (beta=2.0e-3;tadj=4.66) => tau_m ~ 50   ms
	: -> AFTER  (beta=5.0e-5;tadj=4.66) => tau_m ~ 2000 ms
	:
	:also cac was reduced to half, from 1.0 uM to 0.5 uM
}


STATE {
	m
}

ASSIGNED {
	ican	(mA/cm2)
	m_inf
	tau_m	(ms)
	tadj
}

BREAKPOINT { 
	SOLVE states :METHOD euler
	ican = gbar * m*m * (v - ecan)
}

:DERIVATIVE states {
:       evaluate_fct(v,cai)
:
:       m'= (m_inf-m) / tau_m 
:}
  
PROCEDURE states() {
        evaluate_fct(v,cai)
	
        m = m + ( 1-exp(-dt/tau_m) )*(m_inf-m)
	:printf("\n iCAN tau_m=%g", tau_m)

}

UNITSOFF
INITIAL {
:
:  activation kinetics are assumed to be at 22 deg. C
:  Q10 is assumed to be 3
:
	tadj = 3.0 ^ ((celsius-22.0)/10)

	evaluate_fct(v,cai)
	m = m_inf
}


PROCEDURE evaluate_fct(v(mV),cai(mM)) {  LOCAL alpha2

	alpha2 = beta * (cai/cac)^2
	
	tau_m = tau_factor / (alpha2 + beta) / tadj		: tau_m = tau_factor / ( beta * (1 + (cai/cac)^2) ) / tadj
	
	m_inf = alpha2 / (alpha2 + beta)							: m_inf = (cai/cac)^2 / ( 1 + (cai/cac)^2 )

	if(tau_m < taumin) { tau_m = taumin }					: min value of time cst

}
UNITSON
