// Mammalian CA3 cell morphology
//
// Copyright 2007 John L Baker. All rights reserved.
//
// This software is provided AS IS under the terms of the Open Source
// MIT License. See http://www.opensource.org/licenses/mit-license.php.
//
// File: cell_l56a_50_micron.cpp
//
// Release:		1.0.0
// Author:		John Baker
// Updated:		14 July 2006
//
// Description:
//
// Returns CA3 cell morphology in which compartment size is limited 
// to less than 25 microns (approx).
//
// SWC conversion parameters are as follows:
//
//	const float		maxCompLen = 25;	// Max compartment size (a few may be larger)
//	const float		rChgRelTol = 0.25f;	// Fractional change allowed in radius along branch
//	const float		rChgTolLen = 5;		// Minimum branch length when dividing for radius change
//	const float		rMaxForDendrite = 5.0;	// Max dendrite size next to soma (larger are merged into soma)	
//
//	const bool		useYAxisForSomaArea = false;	// Whether to assume soma aligned on Y axis
//	const bool		skipAxon = true;				// Skip writing the axon
//	const bool		debugBranchRadiusChg = false;	// Notify on each change
//
//	const float		maxXJump = 30;		// Maximum jump in x (microns)
//	const float		maxYJump = 30;		// Maximum jump in y (microns)
//	const float		maxZJump = 30;		// Maximum jump in z (microns)
//
// Cell geometry is taken from the Southampton archives.
// This file was created by the utility program swc_reader.
//
// References:
//
// See the Duke/Southamptom Cell Archive at http://neuron.duke.edu/cells/
// for information about SWC files (this was previously located at
// http://www.neuro.soton.ac.uk/cells/). The README file associated
// with the program CVAPP has further information about the SWC format.

#include "bnsf.h"

using namespace BNSF;

// Declare the morphology function as part of the namespace
namespace BAKER_2003 {
	MorphologyEntry* cell_l56a_25_micron();
}

// Provide the body of the function
MorphologyEntry* BAKER_2003::cell_l56a_25_micron() 
{
	
static MorphologyEntry cellMorphology[] = {

// ORIGINAL_SOURCE Neurolucida
// CREATURE rat F344
// REGION Hippocampus
// FIELD/LAYER CA3
// TYPE CA3b Pyramidal Cell in vivo young
// CONTRIBUTOR Buzsaki_G & Turner_DA
// REFERENCE J. Comp. Neurol. 356: 580-594, 1995
// RAW l56a.asc
// EXTRAS Turner_P.CA3
// SOMA_AREA 1.05E3
// SHRINKAGE_CORRECTION 1.33 1.33 2.5
// VERSION_NUMBER 2.0
// VERSION_DATE 1998-03-27
// *********************************************
// SCALE 1.33  1.33  2.5  
// 
{ 1,   0,  -1,   -1,   4.125,  63.473,    0.000,   -0.6,  -1.3,   1.0,   0.0,  0.0,  0.0},
{ 3,   1,   0,   11,   1.441,  19.743,    9.872,   -1.7,  -6.0,  -3.6,  -2.1, -9.4, -9.2},
{ 3,   2,   1,   11,   1.441,  19.743,   29.615,   -3.8, -15.4, -12.8,  -2.1, -9.4, -9.2},
{ 3,   3,   2,   11,   1.441,  19.743,   49.359,   -6.0, -24.8, -21.9,  -2.1, -9.4, -9.2},
{ 3,   4,   3,   18,   1.009,  19.078,   68.769,  -10.5, -33.8, -30.7,  -7.0, -8.6, -8.4},
{ 3,   5,   4,   18,   1.009,  19.078,   87.847,  -17.5, -42.4, -39.1,  -7.0, -8.6, -8.4},
{ 3,   6,   5,   20,   0.720,  17.561,  106.166,  -21.4, -48.0, -34.9,  -0.9, -2.6, 16.8},
{ 3,   7,   6,   42,   0.400,  19.722,  124.808,  -27.9, -49.9, -27.2, -12.0, -1.2, -1.3},
{ 3,   8,   7,   42,   0.400,  19.722,  144.530,  -39.9, -51.1, -28.5, -12.0, -1.2, -1.3},
{ 3,   9,   8,   42,   0.400,  19.722,  164.253,  -51.9, -52.3, -29.8, -12.0, -1.2, -1.3},
{ 3,  10,   9,   42,   0.400,  19.722,  183.975,  -63.9, -53.5, -31.1, -12.0, -1.2, -1.3},
{ 3,  11,  10,   57,   0.247,  23.712,  205.693,  -70.6, -60.0, -31.5,  -1.3,-11.9,  0.4},
{ 3,  12,  11,   57,   0.247,  23.712,  229.405,  -71.9, -71.8, -31.1,  -1.3,-11.9,  0.4},
{ 3,  13,  12,   57,   0.247,  23.712,  253.117,  -73.3, -83.7, -30.7,  -1.3,-11.9,  0.4},
{ 3,  14,  13,   57,   0.247,  23.712,  276.829,  -74.6, -95.6, -30.3,  -1.3,-11.9,  0.4},
{ 3,  15,  14,   57,   0.247,  23.712,  300.540,  -76.0,-107.4, -29.9,  -1.3,-11.9,  0.4},
{ 3,  16,  15,   57,   0.247,  23.712,  324.252,  -77.3,-119.3, -29.5,  -1.3,-11.9,  0.4},
{ 3,  17,   6,   59,   0.560,   6.433,  118.163,  -24.6, -50.7, -26.5,  -5.5, -2.8,  0.0},
{ 3,  18,  17,   67,   0.560,  23.955,  133.358,  -28.3, -60.1, -27.8,  -1.9,-16.1, -2.5},
{ 3,  19,  18,   69,   0.524,  16.258,  153.465,  -32.0, -74.8, -32.0,  -5.5,-13.3, -6.0},
{ 3,  20,  19,   72,   0.400,   8.275,  165.731,  -36.2, -85.2, -35.0,  -3.0, -7.4,  0.0},
{ 3,  21,  20,   75,   0.400,  14.488,  177.113,  -40.2, -94.1, -30.8,  -4.9,-10.5,  8.5},
{ 3,  22,  21,  106,   0.240,  23.926,  196.320,  -43.5,-106.2, -27.2,  -1.8,-13.7, -1.4},
{ 3,  23,  22,  106,   0.240,  23.926,  220.245,  -45.3,-119.9, -28.6,  -1.8,-13.7, -1.4},
{ 3,  24,  23,  106,   0.240,  23.926,  244.171,  -47.1,-133.6, -30.1,  -1.8,-13.7, -1.4},
{ 3,  25,  24,  106,   0.240,  23.926,  268.097,  -48.8,-147.3, -31.5,  -1.8,-13.7, -1.4},
{ 3,  26,  25,  106,   0.240,  23.926,  292.023,  -50.6,-161.0, -32.9,  -1.8,-13.7, -1.4},
{ 3,  27,  26,  106,   0.240,  23.926,  315.949,  -52.4,-174.7, -34.4,  -1.8,-13.7, -1.4},
{ 3,  28,  27,  106,   0.240,  23.926,  339.875,  -54.2,-188.4, -35.8,  -1.8,-13.7, -1.4},
{ 3,  29,  21,  141,   0.240,  22.290,  195.502,  -44.2,-104.6, -28.8,  -3.2,-10.7, -4.5},
{ 3,  30,  29,  141,   0.240,  22.290,  217.791,  -47.5,-115.3, -33.3,  -3.2,-10.7, -4.5},
{ 3,  31,  30,  141,   0.240,  22.290,  240.081,  -50.7,-126.0, -37.8,  -3.2,-10.7, -4.5},
{ 3,  32,  31,  141,   0.240,  22.290,  262.371,  -53.9,-136.6, -42.3,  -3.2,-10.7, -4.5},
{ 3,  33,  32,  141,   0.240,  22.290,  284.661,  -57.2,-147.3, -46.8,  -3.2,-10.7, -4.5},
{ 3,  34,  33,  141,   0.240,  22.290,  306.951,  -60.4,-157.9, -51.3,  -3.2,-10.7, -4.5},
{ 3,  35,  34,  141,   0.240,  22.290,  329.241,  -63.6,-168.6, -55.8,  -3.2,-10.7, -4.5},
{ 3,  36,  20,  144,   0.400,   9.883,  174.810,  -36.4, -92.3, -33.8,   2.8, -7.0,  2.5},
{ 3,  37,  36,  170,   0.245,  22.012,  190.758,  -33.2,-102.9, -35.4,   3.6,-14.1, -5.8},
{ 3,  38,  37,  170,   0.245,  22.012,  212.770,  -29.6,-117.0, -41.1,   3.6,-14.1, -5.8},
{ 3,  39,  38,  170,   0.245,  22.012,  234.782,  -26.0,-131.1, -46.9,   3.6,-14.1, -5.8},
{ 3,  40,  39,  170,   0.245,  22.012,  256.794,  -22.4,-145.2, -52.6,   3.6,-14.1, -5.8},
{ 3,  41,  40,  170,   0.245,  22.012,  278.806,  -18.8,-159.3, -58.4,   3.6,-14.1, -5.8},
{ 3,  42,  41,  170,   0.245,  22.012,  300.818,  -15.2,-173.4, -64.1,   3.6,-14.1, -5.8},
{ 3,  43,  18,  183,   0.240,  16.431,  153.551,  -32.5, -72.9, -28.4,  -6.5, -9.4,  1.3},
{ 3,  44,  43,  183,   0.240,  16.431,  169.982,  -38.9, -82.2, -27.1,  -6.5, -9.4,  1.3},
{ 3,  45,  44,  219,   0.398,  22.724,  189.560,  -44.6, -93.5, -27.8,  -4.9,-13.1, -2.7},
{ 3,  46,  45,  219,   0.398,  22.724,  212.284,  -49.5,-106.6, -30.5,  -4.9,-13.1, -2.7},
{ 3,  47,  46,  219,   0.398,  22.724,  235.008,  -54.4,-119.7, -33.1,  -4.9,-13.1, -2.7},
{ 3,  48,  47,  219,   0.398,  22.724,  257.733,  -59.2,-132.8, -35.8,  -4.9,-13.1, -2.7},
{ 3,  49,  48,  219,   0.398,  22.724,  280.457,  -64.1,-145.9, -38.5,  -4.9,-13.1, -2.7},
{ 3,  50,  49,  219,   0.398,  22.724,  303.181,  -69.0,-159.0, -41.1,  -4.9,-13.1, -2.7},
{ 3,  51,  50,  219,   0.398,  22.724,  325.906,  -73.8,-172.1, -43.8,  -4.9,-13.1, -2.7},
{ 3,  52,  51,  219,   0.398,  22.724,  348.630,  -78.7,-185.2, -46.4,  -4.9,-13.1, -2.7},
{ 3,  53,  17,  222,   0.523,  16.769,  129.765,  -30.8, -55.7, -24.3,  -6.8, -7.2,  4.5},
{ 3,  54,  53,  223,   0.400,   0.488,  138.393,  -34.3, -59.6, -22.0,  -0.2, -0.4,  0.0},
{ 3,  55,  54,  225,   0.400,  15.444,  146.359,  -32.3, -63.4, -15.5,   4.2, -7.2, 13.0},
{ 3,  56,  55,  253,   0.242,  22.832,  165.497,  -32.0, -75.2,  -8.6,  -3.6,-16.6,  0.8},
{ 3,  57,  56,  253,   0.242,  22.832,  188.329,  -35.6, -91.8,  -7.8,  -3.6,-16.6,  0.8},
{ 3,  58,  57,  253,   0.242,  22.832,  211.161,  -39.3,-108.4,  -7.1,  -3.6,-16.6,  0.8},
{ 3,  59,  58,  253,   0.242,  22.832,  233.993,  -42.9,-125.0,  -6.3,  -3.6,-16.6,  0.8},
{ 3,  60,  59,  253,   0.242,  22.832,  256.825,  -46.5,-141.6,  -5.5,  -3.6,-16.6,  0.8},
{ 3,  61,  60,  253,   0.242,  22.832,  279.657,  -50.2,-158.2,  -4.7,  -3.6,-16.6,  0.8},
{ 3,  62,  61,  253,   0.242,  22.832,  302.489,  -53.8,-174.8,  -3.9,  -3.6,-16.6,  0.8},
{ 3,  63,  62,  253,   0.242,  22.832,  325.321,  -57.4,-191.4,  -3.2,  -3.6,-16.6,  0.8},
{ 3,  64,  63,  253,   0.242,  22.832,  348.153,  -61.1,-208.0,  -2.4,  -3.6,-16.6,  0.8},
{ 3,  65,  54,  279,   0.400,  23.480,  150.377,  -38.6, -68.4, -20.3,  -8.4,-17.2,  3.4},
{ 3,  66,  65,  279,   0.400,  23.480,  173.857,  -47.0, -85.6, -17.0,  -8.4,-17.2,  3.4},
{ 3,  67,  66,  279,   0.400,  23.480,  197.337,  -55.4,-102.9, -13.6,  -8.4,-17.2,  3.4},
{ 3,  68,  67,  279,   0.400,  23.480,  220.817,  -63.8,-120.1, -10.3,  -8.4,-17.2,  3.4},
{ 3,  69,  68,  279,   0.400,  23.480,  244.296,  -72.2,-137.4,  -6.9,  -8.4,-17.2,  3.4},
{ 3,  70,  69,  279,   0.400,  23.480,  267.776,  -80.6,-154.6,  -3.6,  -8.4,-17.2,  3.4},
{ 3,  71,  70,  279,   0.400,  23.480,  291.256,  -89.0,-171.9,  -0.2,  -8.4,-17.2,  3.4},
{ 3,  72,   5,  299,   0.240,  23.929,  109.351,  -21.5, -54.2, -37.7,  -1.0,-15.0, 11.2},
{ 3,  73,  72,  299,   0.240,  23.929,  133.280,  -22.4, -69.2, -26.5,  -1.0,-15.0, 11.2},
{ 3,  74,  73,  299,   0.240,  23.929,  157.209,  -23.4, -84.2, -15.3,  -1.0,-15.0, 11.2},
{ 3,  75,  74,  299,   0.240,  23.929,  181.139,  -24.3, -99.2,  -4.1,  -1.0,-15.0, 11.2},
{ 3,  76,  75,  316,   0.240,  24.988,  205.598,  -20.3,-117.4,   0.9,   9.1,-21.4, -1.1},
{ 3,  77,  76,  316,   0.240,  24.988,  230.586,  -11.2,-138.8,  -0.2,   9.1,-21.4, -1.1},
{ 3,  78,  77,  316,   0.240,  24.988,  255.575,   -2.1,-160.2,  -1.3,   9.1,-21.4, -1.1},
{ 3,  79,  78,  316,   0.240,  24.988,  280.563,    6.9,-181.6,  -2.3,   9.1,-21.4, -1.1},
{ 3,  80,  79,  316,   0.240,  24.988,  305.551,   16.0,-203.0,  -3.4,   9.1,-21.4, -1.1},
{ 3,  81,  80,  316,   0.240,  24.988,  330.540,   25.1,-224.4,  -4.5,   9.1,-21.4, -1.1},
{ 3,  82,  75,  323,   0.400,  14.259,  200.233,  -25.4,-112.4,   3.0,  -1.2,-11.4,  3.1},
{ 3,  83,  82,  323,   0.400,  14.259,  214.493,  -26.6,-123.8,   6.1,  -1.2,-11.4,  3.1},
{ 3,  84,  83,  342,   0.247,  24.477,  233.861,  -26.4,-139.2,   6.2,   1.6,-19.4, -3.0},
{ 3,  85,  84,  342,   0.247,  24.477,  258.338,  -24.8,-158.6,   3.3,   1.6,-19.4, -3.0},
{ 3,  86,  85,  342,   0.247,  24.477,  282.814,  -23.3,-177.9,   0.3,   1.6,-19.4, -3.0},
{ 3,  87,  86,  342,   0.247,  24.477,  307.291,  -21.7,-197.3,  -2.7,   1.6,-19.4, -3.0},
{ 3,  88,  87,  342,   0.247,  24.477,  331.768,  -20.2,-216.7,  -5.6,   1.6,-19.4, -3.0},
{ 3,  89,  88,  342,   0.247,  24.477,  356.245,  -18.6,-236.0,  -8.6,   1.6,-19.4, -3.0},
{ 3,  90,   3,  349,   0.784,  20.097,   69.279,   -4.0, -31.3, -32.2,   6.1, -3.7,-11.4},
{ 3,  91,  90,  349,   0.784,  20.097,   89.376,    2.2, -35.0, -43.6,   6.1, -3.7,-11.4},
{ 3,  92,  91,  349,   0.784,  20.097,  109.473,    8.3, -38.7, -55.0,   6.1, -3.7,-11.4},
{ 3,  93,  92,  355,   0.565,  17.091,  128.066,   12.2, -44.3, -62.4,   1.6, -7.5, -3.4},
{ 3,  94,  93,  355,   0.565,  17.091,  145.157,   13.8, -51.8, -65.8,   1.6, -7.5, -3.4},
{ 3,  95,  94,  361,   0.717,  18.497,  162.951,   15.1, -58.3, -64.8,   1.0, -5.4,  5.5},
{ 3,  96,  95,  361,   0.717,  18.497,  181.449,   16.1, -63.6, -59.3,   1.0, -5.4,  5.5},
{ 3,  97,  96,  363,   0.560,  23.083,  202.239,   13.1, -69.4, -66.8,  -7.1, -6.3,-20.5},
{ 3,  98,  97,  381,   0.560,  24.529,  226.045,    8.3, -79.2, -77.9,  -2.5,-13.2, -1.8},
{ 3,  99,  98,  381,   0.560,  24.529,  250.574,    5.9, -92.4, -79.6,  -2.5,-13.2, -1.8},
{ 3, 100,  99,  381,   0.560,  24.529,  275.103,    3.4,-105.6, -81.4,  -2.5,-13.2, -1.8},
{ 3, 101, 100,  381,   0.560,  24.529,  299.632,    0.9,-118.8, -83.1,  -2.5,-13.2, -1.8},
{ 3, 102,  97,  410,   0.400,  24.971,  226.266,   14.9, -77.1, -82.2,  10.7, -9.0,-10.5},
{ 3, 103, 102,  410,   0.400,  24.971,  251.237,   25.6, -86.1, -92.7,  10.7, -9.0,-10.5},
{ 3, 104, 103,  410,   0.400,  24.971,  276.209,   36.3, -95.1,-103.1,  10.7, -9.0,-10.5},
{ 3, 105, 104,  410,   0.400,  24.971,  301.180,   47.0,-104.1,-113.6,  10.7, -9.0,-10.5},
{ 3, 106, 105,  410,   0.400,  24.971,  326.151,   57.7,-113.1,-124.1,  10.7, -9.0,-10.5},
{ 3, 107, 106,  410,   0.400,  24.971,  351.123,   68.4,-122.1,-134.5,  10.7, -9.0,-10.5},
{ 3, 108,  96,  418,   0.560,  17.109,  199.252,   19.3, -68.8, -59.3,   5.2, -4.9, -5.5},
{ 3, 109, 108,  418,   0.560,  17.109,  216.361,   24.5, -73.7, -64.8,   5.2, -4.9, -5.5},
{ 3, 110, 109,  450,   0.400,  23.005,  236.418,   32.6, -80.4, -70.9,  10.9, -8.6, -6.8},
{ 3, 111, 110,  450,   0.400,  23.005,  259.422,   43.4, -89.0, -77.8,  10.9, -8.6, -6.8},
{ 3, 112, 111,  450,   0.400,  23.005,  282.427,   54.3, -97.6, -84.6,  10.9, -8.6, -6.8},
{ 3, 113, 112,  450,   0.400,  23.005,  305.432,   65.1,-106.2, -91.5,  10.9, -8.6, -6.8},
{ 3, 114, 113,  450,   0.400,  23.005,  328.437,   76.0,-114.8, -98.3,  10.9, -8.6, -6.8},
{ 3, 115, 114,  450,   0.400,  23.005,  351.441,   86.9,-123.4,-105.2,  10.9, -8.6, -6.8},
{ 3, 116, 115,  450,   0.400,  23.005,  374.446,   97.7,-131.9,-112.0,  10.9, -8.6, -6.8},
{ 3, 117, 116,  450,   0.400,  23.005,  397.451,  108.6,-140.5,-118.9,  10.9, -8.6, -6.8},
{ 3, 118, 117,  450,   0.400,  23.005,  420.455,  119.4,-149.1,-125.7,  10.9, -8.6, -6.8},
{ 3, 119, 118,  450,   0.400,  23.005,  443.460,  130.3,-157.7,-132.6,  10.9, -8.6, -6.8},
{ 3, 120, 109,  482,   0.400,  23.844,  236.838,   30.8, -81.3, -71.4,   7.2,-10.3, -7.8},
{ 3, 121, 120,  482,   0.400,  23.844,  260.682,   38.0, -91.7, -79.2,   7.2,-10.3, -7.8},
{ 3, 122, 121,  482,   0.400,  23.844,  284.527,   45.2,-102.0, -87.0,   7.2,-10.3, -7.8},
{ 3, 123, 122,  482,   0.400,  23.844,  308.371,   52.5,-112.4, -94.8,   7.2,-10.3, -7.8},
{ 3, 124, 123,  482,   0.400,  23.844,  332.216,   59.7,-122.7,-102.6,   7.2,-10.3, -7.8},
{ 3, 125, 124,  482,   0.400,  23.844,  356.060,   67.0,-133.1,-110.4,   7.2,-10.3, -7.8},
{ 3, 126, 125,  482,   0.400,  23.844,  379.905,   74.2,-143.4,-118.2,   7.2,-10.3, -7.8},
{ 3, 127, 126,  482,   0.400,  23.844,  403.749,   81.5,-153.8,-126.0,   7.2,-10.3, -7.8},
{ 3, 128, 127,  482,   0.400,  23.844,  427.594,   88.7,-164.1,-133.8,   7.2,-10.3, -7.8},
{ 4, 129,   0,  506,   2.150,  13.005,    6.503,  -12.2,  45.1,   8.5,  -1.5,  3.3, 12.5},
{ 4, 130, 129,  509,   2.384,  14.513,   20.262,  -15.9,  51.6,  11.9,  -5.9,  9.6, -5.8},
{ 4, 131, 130,  510,   1.510,   0.925,   27.981,  -19.0,  56.8,   9.1,  -0.2,  0.9,  0.3},
{ 4, 132, 131,  522,   0.901,  22.935,   39.911,  -24.3,  64.3,  12.3, -10.5, 14.1,  6.1},
{ 4, 133, 132,  522,   0.901,  22.935,   62.846,  -34.8,  78.4,  18.4, -10.5, 14.1,  6.1},
{ 4, 134, 133,  522,   0.901,  22.935,   85.781,  -45.3,  92.6,  24.5, -10.5, 14.1,  6.1},
{ 4, 135, 134,  525,   0.880,  18.775,  106.636,  -54.5, 107.6,  29.4,  -7.8, 15.9,  3.8},
{ 4, 136, 135,  535,   0.400,  19.893,  125.969,  -62.4, 122.8,  31.8,  -7.8, 14.5,  1.1},
{ 4, 137, 136,  535,   0.400,  19.893,  145.862,  -70.2, 137.3,  32.9,  -7.8, 14.5,  1.1},
{ 4, 138, 137,  535,   0.400,  19.893,  165.754,  -78.0, 151.8,  34.0,  -7.8, 14.5,  1.1},
{ 4, 139, 135,  538,   0.400,   7.619,  119.832,  -59.9, 115.6,  34.4,  -3.0,  0.2,  6.3},
{ 4, 140, 139,  561,   0.242,  25.212,  136.248,  -65.7, 117.2,  36.7,  -8.5,  2.8, -1.6},
{ 4, 141, 140,  561,   0.242,  25.212,  161.459,  -74.2, 120.0,  35.0,  -8.5,  2.8, -1.6},
{ 4, 142, 141,  561,   0.242,  25.212,  186.671,  -82.7, 122.8,  33.4,  -8.5,  2.8, -1.6},
{ 4, 143, 142,  561,   0.242,  25.212,  211.883,  -91.2, 125.6,  31.7,  -8.5,  2.8, -1.6},
{ 4, 144, 143,  561,   0.242,  25.212,  237.094,  -99.7, 128.5,  30.1,  -8.5,  2.8, -1.6},
{ 4, 145, 134,  564,   0.560,  11.152,  102.824,  -48.4, 100.8,  31.6,   4.4,  2.4,  8.3},
{ 4, 146, 145,  566,   0.696,  14.317,  115.559,  -44.6, 104.2,  38.8,   3.2,  4.4,  6.0},
{ 4, 147, 146,  569,   0.720,  15.796,  130.616,  -41.9, 107.2,  40.8,   2.1,  1.6, -2.0},
{ 4, 148, 147,  576,   0.593,  20.534,  148.781,  -40.9, 112.1,  43.4,  -0.0,  8.1,  7.4},
{ 4, 149, 148,  576,   0.593,  20.534,  169.315,  -40.9, 120.2,  50.8,  -0.0,  8.1,  7.4},
{ 4, 150, 149,  584,   0.560,  24.989,  192.077,  -42.6, 128.4,  58.3,  -3.4,  8.2,  7.5},
{ 4, 151, 150,  584,   0.560,  24.989,  217.065,  -46.0, 136.6,  65.8,  -3.4,  8.2,  7.5},
{ 4, 152, 151,  603,   0.403,  22.427,  240.773,  -47.5, 143.8,  72.6,   0.4,  6.1,  6.1},
{ 4, 153, 152,  603,   0.403,  22.427,  263.200,  -47.1, 149.9,  78.7,   0.4,  6.1,  6.1},
{ 4, 154, 153,  603,   0.403,  22.427,  285.626,  -46.8, 156.1,  84.8,   0.4,  6.1,  6.1},
{ 4, 155, 154,  603,   0.403,  22.427,  308.053,  -46.4, 162.2,  90.9,   0.4,  6.1,  6.1},
{ 4, 156, 149,  609,   0.560,  15.505,  187.335,  -41.8, 131.1,  52.9,  -1.7, 13.7, -3.3},
{ 4, 157, 156,  609,   0.560,  15.505,  202.840,  -43.5, 144.8,  49.6,  -1.7, 13.7, -3.3},
{ 4, 158, 157,  619,   0.400,  18.666,  219.925,  -45.0, 157.2,  52.4,  -1.3, 11.0,  8.8},
{ 4, 159, 158,  619,   0.400,  18.666,  238.591,  -46.3, 168.2,  61.3,  -1.3, 11.0,  8.8},
{ 4, 160, 159,  642,   0.244,  24.998,  260.423,  -48.4, 183.5,  70.0,  -2.7, 19.6,  8.6},
{ 4, 161, 160,  642,   0.244,  24.998,  285.421,  -51.1, 203.2,  78.6,  -2.7, 19.6,  8.6},
{ 4, 162, 161,  642,   0.244,  24.998,  310.419,  -53.8, 222.8,  87.3,  -2.7, 19.6,  8.6},
{ 4, 163, 162,  642,   0.244,  24.998,  335.417,  -56.5, 242.4,  95.9,  -2.7, 19.6,  8.6},
{ 4, 164, 157,  674,   0.400,  22.223,  221.703,  -41.4, 157.4,  51.9,   5.9, 11.5,  7.9},
{ 4, 165, 164,  674,   0.400,  22.223,  243.926,  -35.5, 168.9,  59.8,   5.9, 11.5,  7.9},
{ 4, 166, 165,  674,   0.400,  22.223,  266.149,  -29.6, 180.4,  67.7,   5.9, 11.5,  7.9},
{ 4, 167, 166,  674,   0.400,  22.223,  288.372,  -23.7, 191.9,  75.6,   5.9, 11.5,  7.9},
{ 4, 168, 167,  674,   0.400,  22.223,  310.595,  -17.8, 203.5,  83.4,   5.9, 11.5,  7.9},
{ 4, 169, 168,  674,   0.400,  22.223,  332.818,  -11.9, 215.0,  91.3,   5.9, 11.5,  7.9},
{ 4, 170, 147,  689,   0.240,  22.856,  149.942,  -43.5, 111.6,  43.7,  -5.3,  7.1,  7.9},
{ 4, 171, 170,  689,   0.240,  22.856,  172.798,  -48.8, 118.7,  51.7,  -5.3,  7.1,  7.9},
{ 4, 172, 171,  689,   0.240,  22.856,  195.655,  -54.1, 125.9,  59.6,  -5.3,  7.1,  7.9},
{ 4, 173, 172,  689,   0.240,  22.856,  218.511,  -59.4, 133.0,  67.5,  -5.3,  7.1,  7.9},
{ 4, 174, 173,  721,   0.400,  25.297,  242.588,  -67.5, 140.3,  75.2, -10.9,  7.4,  7.4},
{ 4, 175, 174,  721,   0.400,  25.297,  267.884,  -78.5, 147.8,  82.6, -10.9,  7.4,  7.4},
{ 4, 176, 175,  721,   0.400,  25.297,  293.181,  -89.4, 155.2,  90.0, -10.9,  7.4,  7.4},
{ 4, 177, 176,  721,   0.400,  25.297,  318.478, -100.4, 162.6,  97.4, -10.9,  7.4,  7.4},
{ 4, 178, 177,  721,   0.400,  25.297,  343.774, -111.3, 170.1, 104.8, -10.9,  7.4,  7.4},
{ 4, 179, 173,  723,   0.318,   6.714,  233.296,  -61.5, 138.2,  68.8,   1.1,  3.2, -5.5},
{ 4, 180, 179,  759,   0.400,  22.907,  248.107,  -61.8, 145.4,  68.7,  -1.5, 11.1,  5.4},
{ 4, 181, 180,  759,   0.400,  22.907,  271.014,  -63.3, 156.4,  74.1,  -1.5, 11.1,  5.4},
{ 4, 182, 181,  759,   0.400,  22.907,  293.921,  -64.8, 167.5,  79.6,  -1.5, 11.1,  5.4},
{ 4, 183, 182,  759,   0.400,  22.907,  316.828,  -66.3, 178.6,  85.0,  -1.5, 11.1,  5.4},
{ 4, 184, 183,  759,   0.400,  22.907,  339.735,  -67.8, 189.7,  90.4,  -1.5, 11.1,  5.4},
{ 4, 185, 184,  759,   0.400,  22.907,  362.643,  -69.3, 200.7,  95.9,  -1.5, 11.1,  5.4},
{ 4, 186, 185,  759,   0.400,  22.907,  385.550,  -70.8, 211.8, 101.3,  -1.5, 11.1,  5.4},
{ 4, 187, 186,  759,   0.400,  22.907,  408.457,  -72.3, 222.9, 106.7,  -1.5, 11.1,  5.4},
{ 4, 188, 146,  762,   0.481,  10.520,  127.978,  -44.8, 109.4,  41.4,  -3.6,  5.9, -0.8},
{ 4, 189, 188,  764,   0.560,  12.372,  139.424,  -48.5, 115.2,  42.6,  -3.7,  5.9,  3.3},
{ 4, 190, 189,  770,   0.560,  27.247,  159.233,  -54.7, 129.7,  43.6,  -8.6, 23.0, -1.3},
{ 4, 191, 190,  815,   0.400,  24.162,  184.937,  -63.8, 147.7,  39.3,  -9.6, 12.9, -7.3},
{ 4, 192, 191,  815,   0.400,  24.162,  209.099,  -73.3, 160.6,  32.0,  -9.6, 12.9, -7.3},
{ 4, 193, 192,  815,   0.400,  24.162,  233.260,  -82.9, 173.5,  24.7,  -9.6, 12.9, -7.3},
{ 4, 194, 193,  815,   0.400,  24.162,  257.422,  -92.4, 186.4,  17.4,  -9.6, 12.9, -7.3},
{ 4, 195, 194,  815,   0.400,  24.162,  281.584, -102.0, 199.3,  10.1,  -9.6, 12.9, -7.3},
{ 4, 196, 195,  815,   0.400,  24.162,  305.745, -111.6, 212.2,   2.8,  -9.6, 12.9, -7.3},
{ 4, 197, 196,  815,   0.400,  24.162,  329.907, -121.1, 225.1,  -4.5,  -9.6, 12.9, -7.3},
{ 4, 198, 197,  815,   0.400,  24.162,  354.068, -130.7, 238.0, -11.8,  -9.6, 12.9, -7.3},
{ 4, 199, 198,  815,   0.400,  24.162,  378.230, -140.3, 251.0, -19.1,  -9.6, 12.9, -7.3},
{ 4, 200, 190,  826,   0.560,  21.906,  183.810,  -62.5, 149.4,  41.6,  -7.0, 16.4, -2.8},
{ 4, 201, 200,  826,   0.560,  21.906,  205.716,  -69.5, 165.8,  38.9,  -7.0, 16.4, -2.8},
{ 4, 202, 201,  826,   0.560,  21.906,  227.622,  -76.6, 182.2,  36.1,  -7.0, 16.4, -2.8},
{ 4, 203, 202,  826,   0.560,  21.906,  249.528,  -83.6, 198.6,  33.4,  -7.0, 16.4, -2.8},
{ 4, 204, 203,  829,   0.560,  14.024,  267.493,  -87.3, 213.8,  32.0,  -0.2, 14.0,  0.0},
{ 4, 205, 204,  857,   0.240,  22.691,  285.851,  -91.4, 226.3,  31.7,  -8.0, 10.9, -0.7},
{ 4, 206, 205,  857,   0.240,  22.691,  308.542,  -99.4, 237.2,  31.0,  -8.0, 10.9, -0.7},
{ 4, 207, 206,  857,   0.240,  22.691,  331.234, -107.4, 248.1,  30.4,  -8.0, 10.9, -0.7},
{ 4, 208, 207,  857,   0.240,  22.691,  353.925, -115.4, 259.1,  29.7,  -8.0, 10.9, -0.7},
{ 4, 209, 208,  857,   0.240,  22.691,  376.616, -123.4, 270.0,  29.0,  -8.0, 10.9, -0.7},
{ 4, 210, 209,  857,   0.240,  22.691,  399.307, -131.4, 281.0,  28.4,  -8.0, 10.9, -0.7},
{ 4, 211, 210,  857,   0.240,  22.691,  421.999, -139.4, 291.9,  27.7,  -8.0, 10.9, -0.7},
{ 4, 212, 211,  857,   0.240,  22.691,  444.690, -147.4, 302.8,  27.1,  -8.0, 10.9, -0.7},
{ 4, 213, 204,  860,   0.560,  18.766,  283.889,  -85.8, 229.8,  32.0,   3.1, 17.9,  0.0},
{ 4, 214, 213,  863,   0.684,  20.315,  303.429,  -83.2, 248.7,  32.0,   2.2, 20.0,  0.0},
{ 4, 215, 214,  908,   0.400,  23.618,  325.395,  -84.8, 266.0,  31.8,  -5.5, 14.6, -0.5},
{ 4, 216, 215,  908,   0.400,  23.618,  349.013,  -90.2, 280.6,  31.3,  -5.5, 14.6, -0.5},
{ 4, 217, 216,  908,   0.400,  23.618,  372.632,  -95.7, 295.2,  30.8,  -5.5, 14.6, -0.5},
{ 4, 218, 217,  908,   0.400,  23.618,  396.250, -101.1, 309.8,  30.3,  -5.5, 14.6, -0.5},
{ 4, 219, 218,  908,   0.400,  23.618,  419.868, -106.6, 324.4,  29.8,  -5.5, 14.6, -0.5},
{ 4, 220, 219,  908,   0.400,  23.618,  443.486, -112.1, 339.0,  29.3,  -5.5, 14.6, -0.5},
{ 4, 221, 220,  908,   0.400,  23.618,  467.104, -117.5, 353.6,  28.8,  -5.5, 14.6, -0.5},
{ 4, 222, 221,  908,   0.400,  23.618,  490.722, -123.0, 368.2,  28.3,  -5.5, 14.6, -0.5},
{ 4, 223, 222,  908,   0.400,  23.618,  514.340, -128.4, 382.8,  27.8,  -5.5, 14.6, -0.5},
{ 4, 224, 223,  908,   0.400,  23.618,  537.958, -133.9, 397.4,  27.3,  -5.5, 14.6, -0.5},
{ 4, 225, 224,  924,   0.240,  18.850,  559.192, -134.9, 411.4,  26.1,   3.5, 13.5, -1.9},
{ 4, 226, 225,  924,   0.240,  18.850,  578.042, -131.4, 424.9,  24.2,   3.5, 13.5, -1.9},
{ 4, 227, 224,  939,   0.240,  22.937,  561.235, -140.2, 414.5,  27.8,  -7.3, 19.7,  1.6},
{ 4, 228, 227,  939,   0.240,  22.937,  584.173, -147.5, 434.2,  29.4,  -7.3, 19.7,  1.6},
{ 4, 229, 228,  939,   0.240,  22.937,  607.110, -154.8, 453.9,  31.1,  -7.3, 19.7,  1.6},
{ 4, 230, 229,  939,   0.240,  22.937,  630.047, -162.1, 473.6,  32.7,  -7.3, 19.7,  1.6},
{ 4, 231, 214,  967,   0.565,  25.313,  326.243,  -79.6, 269.1,  32.7,   5.0, 20.8,  1.5},
{ 4, 232, 231,  967,   0.565,  25.313,  351.556,  -74.6, 290.0,  34.2,   5.0, 20.8,  1.5},
{ 4, 233, 232,  967,   0.565,  25.313,  376.870,  -69.6, 310.8,  35.6,   5.0, 20.8,  1.5},
{ 4, 234, 233,  967,   0.565,  25.313,  402.183,  -64.6, 331.6,  37.1,   5.0, 20.8,  1.5},
{ 4, 235, 234,  967,   0.565,  25.313,  427.497,  -59.7, 352.4,  38.5,   5.0, 20.8,  1.5},
{ 4, 236, 235, 1005,   0.400,  22.537,  451.422,  -55.3, 370.1,  39.1,   3.7, 14.4, -0.2},
{ 4, 237, 236, 1005,   0.400,  22.537,  473.959,  -51.6, 384.5,  38.9,   3.7, 14.4, -0.2},
{ 4, 238, 237, 1005,   0.400,  22.537,  496.496,  -47.8, 398.9,  38.7,   3.7, 14.4, -0.2},
{ 4, 239, 238, 1005,   0.400,  22.537,  519.033,  -44.1, 413.3,  38.5,   3.7, 14.4, -0.2},
{ 4, 240, 239, 1005,   0.400,  22.537,  541.571,  -40.4, 427.7,  38.3,   3.7, 14.4, -0.2},
{ 4, 241, 240, 1005,   0.400,  22.537,  564.108,  -36.7, 442.1,  38.0,   3.7, 14.4, -0.2},
{ 4, 242, 241, 1005,   0.400,  22.537,  586.645,  -32.9, 456.5,  37.8,   3.7, 14.4, -0.2},
{ 4, 243, 242, 1005,   0.400,  22.537,  609.182,  -29.2, 470.9,  37.6,   3.7, 14.4, -0.2},
{ 4, 244, 243, 1005,   0.400,  22.537,  631.719,  -25.5, 485.3,  37.4,   3.7, 14.4, -0.2},
{ 4, 245, 235, 1046,   0.560,  24.208,  452.257,  -57.5, 368.3,  37.9,  -0.8, 11.0, -2.8},
{ 4, 246, 245, 1046,   0.560,  24.208,  476.465,  -58.3, 379.3,  35.1,  -0.8, 11.0, -2.8},
{ 4, 247, 246, 1046,   0.560,  24.208,  500.673,  -59.0, 390.3,  32.4,  -0.8, 11.0, -2.8},
{ 4, 248, 247, 1046,   0.560,  24.208,  524.882,  -59.8, 401.3,  29.6,  -0.8, 11.0, -2.8},
{ 4, 249, 248, 1046,   0.560,  24.208,  549.090,  -60.5, 412.3,  26.9,  -0.8, 11.0, -2.8},
{ 4, 250, 249, 1046,   0.560,  24.208,  573.298,  -61.3, 423.3,  24.1,  -0.8, 11.0, -2.8},
{ 4, 251, 250, 1046,   0.560,  24.208,  597.506,  -62.1, 434.3,  21.4,  -0.8, 11.0, -2.8},
{ 4, 252, 203, 1056,   0.400,  22.662,  271.812,  -91.1, 208.5,  34.2,  -7.9,  3.3,  4.4},
{ 4, 253, 252, 1056,   0.400,  22.662,  294.474,  -99.1, 211.8,  38.6,  -7.9,  3.3,  4.4},
{ 4, 254, 253, 1056,   0.400,  22.662,  317.136, -107.0, 215.2,  42.9,  -7.9,  3.3,  4.4},
{ 4, 255, 254, 1056,   0.400,  22.662,  339.797, -114.9, 218.5,  47.3,  -7.9,  3.3,  4.4},
{ 4, 256, 255, 1066,   0.240,  19.145,  360.701, -125.6, 226.1,  47.4, -13.4, 11.8, -4.3},
{ 4, 257, 256, 1066,   0.240,  19.145,  379.845, -139.0, 237.9,  43.1, -13.4, 11.8, -4.3},
{ 4, 258, 255, 1068,   0.323,   6.837,  354.547, -122.2, 220.3,  49.5,  -6.7,  0.3,  0.0},
{ 4, 259, 258, 1074,   0.240,  17.987,  366.959, -133.5, 222.8,  48.1, -15.9,  4.7, -2.9},
{ 4, 260, 259, 1074,   0.240,  17.987,  384.945, -149.5, 227.5,  45.2, -15.9,  4.7, -2.9},
{ 4, 261, 189, 1081,   0.560,  16.021,  153.620,  -53.9, 120.4,  40.3,  -7.1,  4.3, -7.9},
{ 4, 262, 261, 1081,   0.560,  16.021,  169.641,  -61.0, 124.7,  32.4,  -7.1,  4.3, -7.9},
{ 4, 263, 262, 1116,   0.240,  23.349,  189.326,  -70.6, 128.7,  28.2, -12.2,  3.6, -0.6},
{ 4, 264, 263, 1116,   0.240,  23.349,  212.676,  -82.8, 132.3,  27.6, -12.2,  3.6, -0.6},
{ 4, 265, 264, 1116,   0.240,  23.349,  236.025,  -95.0, 135.9,  27.0, -12.2,  3.6, -0.6},
{ 4, 266, 265, 1116,   0.240,  23.349,  259.374, -107.2, 139.5,  26.4, -12.2,  3.6, -0.6},
{ 4, 267, 266, 1116,   0.240,  23.349,  282.724, -119.4, 143.1,  25.8, -12.2,  3.6, -0.6},
{ 4, 268, 267, 1116,   0.240,  23.349,  306.073, -131.6, 146.7,  25.1, -12.2,  3.6, -0.6},
{ 4, 269, 268, 1116,   0.240,  23.349,  329.423, -143.9, 150.3,  24.5, -12.2,  3.6, -0.6},
{ 4, 270, 269, 1116,   0.240,  23.349,  352.772, -156.1, 153.9,  23.9, -12.2,  3.6, -0.6},
{ 4, 271, 270, 1116,   0.240,  23.349,  376.122, -168.3, 157.5,  23.3, -12.2,  3.6, -0.6},
{ 4, 272, 262, 1118,   0.483,  18.157,  186.730,  -64.2, 128.0,  36.8,   0.6,  2.3, 16.5},
{ 4, 273, 272, 1141,   0.248,  19.883,  205.750,  -67.8, 131.2,  42.6,  -7.6,  4.2, -4.8},
{ 4, 274, 273, 1141,   0.248,  19.883,  225.633,  -75.4, 135.5,  37.8,  -7.6,  4.2, -4.8},
{ 4, 275, 274, 1141,   0.248,  19.883,  245.516,  -83.0, 139.7,  33.0,  -7.6,  4.2, -4.8},
{ 4, 276, 275, 1141,   0.248,  19.883,  265.399,  -90.6, 143.9,  28.2,  -7.6,  4.2, -4.8},
{ 4, 277, 131, 1158,   0.716,  24.133,   40.510,  -19.2,  66.4,  10.9,  -0.3, 18.4,  3.3},
{ 4, 278, 277, 1158,   0.716,  24.133,   64.643,  -19.5,  84.9,  14.1,  -0.3, 18.4,  3.3},
{ 4, 279, 278, 1158,   0.716,  24.133,   88.775,  -19.8, 103.3,  17.4,  -0.3, 18.4,  3.3},
{ 4, 280, 279, 1158,   0.716,  24.133,  112.908,  -20.2, 121.7,  20.6,  -0.3, 18.4,  3.3},
{ 4, 281, 280, 1174,   0.400,  22.999,  136.474,  -19.7, 137.1,  24.0,   1.2, 12.3,  3.5},
{ 4, 282, 281, 1174,   0.400,  22.999,  159.473,  -18.5, 149.5,  27.5,   1.2, 12.3,  3.5},
{ 4, 283, 282, 1174,   0.400,  22.999,  182.472,  -17.3, 161.8,  31.0,   1.2, 12.3,  3.5},
{ 4, 284, 283, 1174,   0.400,  22.999,  205.470,  -16.1, 174.2,  34.5,   1.2, 12.3,  3.5},
{ 4, 285, 284, 1183,   0.248,  15.670,  224.805,  -12.2, 184.4,  33.7,   6.5,  8.1, -5.1},
{ 4, 286, 285, 1183,   0.248,  15.670,  240.474,   -5.8, 192.4,  28.6,   6.5,  8.1, -5.1},
{ 4, 287, 286, 1198,   0.240,  20.792,  258.705,   -4.2, 199.8,  26.6,  -3.4,  6.7,  1.2},
{ 4, 288, 287, 1198,   0.240,  20.792,  279.497,   -7.6, 206.6,  27.8,  -3.4,  6.7,  1.2},
{ 4, 289, 288, 1198,   0.240,  20.792,  300.289,  -11.0, 213.3,  28.9,  -3.4,  6.7,  1.2},
{ 4, 290, 289, 1201,   0.240,   7.567,  314.468,  -13.7, 216.9,  25.9,  -1.9,  0.4, -7.3},
{ 4, 291, 289, 1235,   0.240,  22.467,  321.918,  -12.1, 224.4,  28.6,   1.3, 15.5, -1.8},
{ 4, 292, 291, 1235,   0.240,  22.467,  344.385,  -10.8, 239.9,  26.9,   1.3, 15.5, -1.8},
{ 4, 293, 292, 1235,   0.240,  22.467,  366.852,   -9.5, 255.4,  25.1,   1.3, 15.5, -1.8},
{ 4, 294, 293, 1235,   0.240,  22.467,  389.319,   -8.3, 270.8,  23.4,   1.3, 15.5, -1.8},
{ 4, 295, 294, 1235,   0.240,  22.467,  411.786,   -7.0, 286.3,  21.6,   1.3, 15.5, -1.8},
{ 4, 296, 295, 1235,   0.240,  22.467,  434.253,   -5.7, 301.8,  19.9,   1.3, 15.5, -1.8},
{ 4, 297, 296, 1235,   0.240,  22.467,  456.720,   -4.4, 317.3,  18.1,   1.3, 15.5, -1.8},
{ 4, 298, 297, 1235,   0.240,  22.467,  479.187,   -3.2, 332.7,  16.4,   1.3, 15.5, -1.8},
{ 4, 299, 286, 1243,   0.240,  20.155,  258.387,    0.1, 205.0,  27.4,   5.3, 17.2,  2.8},
{ 4, 300, 299, 1243,   0.240,  20.155,  278.542,    5.4, 222.2,  30.1,   5.3, 17.2,  2.8},
{ 4, 301, 300, 1262,   0.240,  22.429,  299.834,    8.1, 240.6,  30.3,   0.1, 19.5, -2.3},
{ 4, 302, 301, 1262,   0.240,  22.429,  322.263,    8.2, 260.1,  28.0,   0.1, 19.5, -2.3},
{ 4, 303, 302, 1262,   0.240,  22.429,  344.692,    8.3, 279.6,  25.7,   0.1, 19.5, -2.3},
{ 4, 304, 303, 1262,   0.240,  22.429,  367.122,    8.4, 299.2,  23.4,   0.1, 19.5, -2.3},
{ 4, 305, 300, 1272,   0.240,  24.039,  300.639,    7.2, 236.4,  32.3,  -1.7, 11.2,  1.6},
{ 4, 306, 305, 1272,   0.240,  24.039,  324.677,    5.4, 247.6,  34.0,  -1.7, 11.2,  1.6},
{ 4, 307, 306, 1272,   0.240,  24.039,  348.716,    3.7, 258.9,  35.6,  -1.7, 11.2,  1.6},
{ 4, 308, 307, 1272,   0.240,  24.039,  372.754,    2.0, 270.1,  37.3,  -1.7, 11.2,  1.6},
{ 4, 309, 308, 1272,   0.240,  24.039,  396.793,    0.2, 281.3,  38.9,  -1.7, 11.2,  1.6},
{ 4, 310, 280, 1283,   0.400,  20.087,  135.018,  -19.1, 139.6,  21.1,   2.4, 17.2, -2.3},
{ 4, 311, 310, 1283,   0.400,  20.087,  155.105,  -16.8, 156.7,  18.9,   2.4, 17.2, -2.3},
{ 4, 312, 311, 1283,   0.400,  20.087,  175.191,  -14.4, 173.9,  16.6,   2.4, 17.2, -2.3},
{ 4, 313, 312, 1283,   0.400,  20.087,  195.278,  -12.0, 191.1,  14.4,   2.4, 17.2, -2.3},
{ 4, 314, 313, 1294,   0.400,  22.128,  216.386,  -12.6, 205.4,  16.3,  -3.6, 11.4,  6.1},
{ 4, 315, 314, 1294,   0.400,  22.128,  238.514,  -16.2, 216.8,  22.4,  -3.6, 11.4,  6.1},
{ 4, 316, 315, 1294,   0.400,  22.128,  260.643,  -19.8, 228.2,  28.5,  -3.6, 11.4,  6.1},
{ 4, 317, 316, 1297,   0.533,  15.716,  279.565,  -14.0, 235.1,  31.1,  15.3,  2.5, -0.8},
{ 4, 318, 317, 1300,   0.526,  20.024,  297.435,   -5.4, 243.5,  35.0,   1.9, 14.3,  8.5},
{ 4, 319, 318, 1305,   0.560,  20.276,  317.584,   -4.7, 257.0,  36.6,  -0.5, 12.7, -5.4},
{ 4, 320, 319, 1305,   0.560,  20.276,  337.860,   -5.2, 269.6,  31.2,  -0.5, 12.7, -5.4},
{ 4, 321, 320, 1313,   0.407,  20.269,  358.133,   -6.0, 284.2,  30.3,  -1.1, 16.4,  3.5},
{ 4, 322, 321, 1313,   0.407,  20.269,  378.401,   -7.1, 300.5,  33.8,  -1.1, 16.4,  3.5},
{ 4, 323, 322, 1313,   0.407,  20.269,  398.670,   -8.2, 316.9,  37.3,  -1.1, 16.4,  3.5},
{ 4, 324, 323, 1327,   0.400,  23.007,  420.308,   -6.5, 329.6,  41.4,   4.5,  9.0,  4.9},
{ 4, 325, 324, 1327,   0.400,  23.007,  443.314,   -2.0, 338.7,  46.3,   4.5,  9.0,  4.9},
{ 4, 326, 325, 1379,   0.244,  24.129,  466.882,    1.4, 350.6,  50.5,   2.2, 14.8,  3.5},
{ 4, 327, 326, 1379,   0.244,  24.129,  491.011,    3.6, 365.3,  54.0,   2.2, 14.8,  3.5},
{ 4, 328, 327, 1379,   0.244,  24.129,  515.140,    5.8, 380.1,  57.5,   2.2, 14.8,  3.5},
{ 4, 329, 328, 1379,   0.244,  24.129,  539.269,    7.9, 394.9,  61.0,   2.2, 14.8,  3.5},
{ 4, 330, 329, 1379,   0.244,  24.129,  563.398,   10.1, 409.6,  64.5,   2.2, 14.8,  3.5},
{ 4, 331, 330, 1379,   0.244,  24.129,  587.527,   12.3, 424.4,  68.0,   2.2, 14.8,  3.5},
{ 4, 332, 331, 1379,   0.244,  24.129,  611.656,   14.5, 439.1,  71.5,   2.2, 14.8,  3.5},
{ 4, 333, 332, 1379,   0.244,  24.129,  635.785,   16.7, 453.9,  75.0,   2.2, 14.8,  3.5},
{ 4, 334, 333, 1379,   0.244,  24.129,  659.914,   18.8, 468.7,  78.5,   2.2, 14.8,  3.5},
{ 4, 335, 323, 1390,   0.400,  19.303,  418.456,   -9.3, 331.6,  43.1,  -1.1, 13.0,  8.3},
{ 4, 336, 335, 1390,   0.400,  19.303,  437.759,  -10.4, 344.6,  51.4,  -1.1, 13.0,  8.3},
{ 4, 337, 336, 1390,   0.400,  19.303,  457.062,  -11.6, 357.6,  59.6,  -1.1, 13.0,  8.3},
{ 4, 338, 337, 1415,   0.244,  23.236,  478.331,  -15.3, 371.4,  66.7,  -6.3, 14.4,  5.9},
{ 4, 339, 338, 1415,   0.244,  23.236,  501.568,  -21.6, 385.8,  72.6,  -6.3, 14.4,  5.9},
{ 4, 340, 339, 1415,   0.244,  23.236,  524.804,  -27.9, 400.2,  78.5,  -6.3, 14.4,  5.9},
{ 4, 341, 340, 1415,   0.244,  23.236,  548.041,  -34.2, 414.6,  84.4,  -6.3, 14.4,  5.9},
{ 4, 342, 341, 1415,   0.244,  23.236,  571.277,  -40.6, 429.0,  90.3,  -6.3, 14.4,  5.9},
{ 4, 343, 342, 1415,   0.244,  23.236,  594.514,  -46.9, 443.4,  96.2,  -6.3, 14.4,  5.9},
{ 4, 344, 343, 1415,   0.244,  23.236,  617.750,  -53.2, 457.8, 102.1,  -6.3, 14.4,  5.9},
{ 4, 345, 313, 1451,   0.240,  24.366,  217.505,  -10.3, 208.9,  13.7,   1.1, 18.3,  0.8},
{ 4, 346, 345, 1451,   0.240,  24.366,  241.871,   -9.2, 227.2,  14.5,   1.1, 18.3,  0.8},
{ 4, 347, 346, 1451,   0.240,  24.366,  266.237,   -8.1, 245.5,  15.4,   1.1, 18.3,  0.8},
{ 4, 348, 347, 1451,   0.240,  24.366,  290.603,   -7.0, 263.8,  16.2,   1.1, 18.3,  0.8},
{ 4, 349, 348, 1451,   0.240,  24.366,  314.969,   -5.9, 282.1,  17.0,   1.1, 18.3,  0.8},
{ 4, 350, 349, 1451,   0.240,  24.366,  339.335,   -4.8, 300.4,  17.9,   1.1, 18.3,  0.8},
{ 4, 351, 350, 1451,   0.240,  24.366,  363.701,   -3.7, 318.7,  18.7,   1.1, 18.3,  0.8},
{ 4, 352, 351, 1451,   0.240,  24.366,  388.067,   -2.7, 337.0,  19.6,   1.1, 18.3,  0.8},
{ 4, 353, 129, 1455,   1.621,  21.144,   23.577,  -11.6,  55.4,  16.8,   2.5, 17.2,  4.0},
{ 4, 354, 353, 1467,   1.195,  22.865,   45.582,   -9.3,  73.7,  15.8,   2.0, 19.4, -6.0},
{ 4, 355, 354, 1467,   1.195,  22.865,   68.447,   -7.3,  93.1,   9.8,   2.0, 19.4, -6.0},
{ 4, 356, 355, 1467,   1.195,  22.865,   91.313,   -5.3, 112.5,   3.8,   2.0, 19.4, -6.0},
{ 4, 357, 356, 1470,   0.400,  11.419,  108.455,   -0.5, 124.1,   4.1,   7.4,  3.7,  6.7},
{ 4, 358, 357, 1487,   0.400,  21.195,  124.762,    7.5, 127.2,   6.4,   8.7,  2.4, -2.0},
{ 4, 359, 358, 1487,   0.400,  21.195,  145.957,   16.2, 129.6,   4.4,   8.7,  2.4, -2.0},
{ 4, 360, 359, 1487,   0.400,  21.195,  167.152,   24.9, 131.9,   2.4,   8.7,  2.4, -2.0},
{ 4, 361, 360, 1502,   0.400,  21.512,  188.505,   35.3, 136.1,  -2.4,  12.1,  5.9, -7.7},
{ 4, 362, 361, 1502,   0.400,  21.512,  210.017,   47.3, 142.0, -10.0,  12.1,  5.9, -7.7},
{ 4, 363, 362, 1502,   0.400,  21.512,  231.529,   59.4, 147.8, -17.7,  12.1,  5.9, -7.7},
{ 4, 364, 363, 1515,   0.240,  21.656,  253.113,   74.8, 149.9, -22.2,  18.6, -1.7, -1.3},
{ 4, 365, 364, 1515,   0.240,  21.656,  274.769,   93.4, 148.2, -23.5,  18.6, -1.7, -1.3},
{ 4, 366, 365, 1515,   0.240,  21.656,  296.425,  112.0, 146.4, -24.9,  18.6, -1.7, -1.3},
{ 4, 367, 363, 1519,   0.400,  27.042,  255.806,   68.6, 155.4, -12.2,   6.3,  9.4, 18.8},
{ 4, 368, 367, 1552,   0.250,  23.088,  280.871,   76.9, 165.6,  -6.2,  10.2, 10.9, -6.8},
{ 4, 369, 368, 1552,   0.250,  23.088,  303.959,   87.0, 176.5, -13.1,  10.2, 10.9, -6.8},
{ 4, 370, 369, 1552,   0.250,  23.088,  327.047,   97.2, 187.4, -19.9,  10.2, 10.9, -6.8},
{ 4, 371, 370, 1552,   0.250,  23.088,  350.135,  107.4, 198.3, -26.8,  10.2, 10.9, -6.8},
{ 4, 372, 371, 1552,   0.250,  23.088,  373.223,  117.5, 209.3, -33.6,  10.2, 10.9, -6.8},
{ 4, 373, 360, 1558,   0.400,  21.480,  188.490,   35.7, 139.5,   0.3,  12.9, 12.6, -2.3},
{ 4, 374, 373, 1600,   0.243,  24.001,  211.230,   50.0, 147.7,  -1.7,  15.6,  3.9, -1.8},
{ 4, 375, 374, 1600,   0.243,  24.001,  235.231,   65.6, 151.5,  -3.5,  15.6,  3.9, -1.8},
{ 4, 376, 375, 1600,   0.243,  24.001,  259.233,   81.2, 155.4,  -5.4,  15.6,  3.9, -1.8},
{ 4, 377, 376, 1600,   0.243,  24.001,  283.234,   96.9, 159.2,  -7.2,  15.6,  3.9, -1.8},
{ 4, 378, 377, 1600,   0.243,  24.001,  307.235,  112.5, 163.1,  -9.0,  15.6,  3.9, -1.8},
{ 4, 379, 378, 1600,   0.243,  24.001,  331.236,  128.1, 167.0, -10.8,  15.6,  3.9, -1.8},
{ 4, 380, 379, 1600,   0.243,  24.001,  355.237,  143.8, 170.8, -12.6,  15.6,  3.9, -1.8},
{ 4, 381, 357, 1602,   0.400,   3.152,  115.740,    2.3, 127.0,   6.8,  -1.7,  2.2, -1.3},
{ 4, 382, 381, 1620,   0.240,  24.224,  129.428,    8.4, 126.9,   3.2,  13.8, -2.6, -6.0},
{ 4, 383, 382, 1620,   0.240,  24.224,  153.652,   22.2, 124.3,  -2.8,  13.8, -2.6, -6.0},
{ 4, 384, 383, 1620,   0.240,  24.224,  177.876,   36.0, 121.7,  -8.8,  13.8, -2.6, -6.0},
{ 4, 385, 384, 1620,   0.240,  24.224,  202.100,   49.8, 119.2, -14.8,  13.8, -2.6, -6.0},
{ 4, 386, 381, 1622,   0.400,   2.965,  118.799,    0.5, 127.9,   6.9,  -1.9, -0.4,  1.5},
{ 4, 387, 386, 1631,   0.240,  18.940,  129.751,    3.5, 131.5,   3.0,   7.8,  7.6, -9.3},
{ 4, 388, 387, 1631,   0.240,  18.940,  148.692,   11.2, 139.1,  -6.3,   7.8,  7.6, -9.3},
{ 4, 389, 388, 1631,   0.240,  18.940,  167.632,   19.0, 146.7, -15.6,   7.8,  7.6, -9.3},
{ 4, 390, 389, 1649,   0.392,  24.097,  189.151,   28.8, 155.1, -19.8,  11.8,  9.0,  0.9},
{ 4, 391, 390, 1649,   0.392,  24.097,  213.247,   40.6, 164.1, -18.9,  11.8,  9.0,  0.9},
{ 4, 392, 391, 1649,   0.392,  24.097,  237.344,   52.4, 173.1, -17.9,  11.8,  9.0,  0.9},
{ 4, 393, 392, 1649,   0.392,  24.097,  261.440,   64.2, 182.1, -17.0,  11.8,  9.0,  0.9},
{ 4, 394, 393, 1649,   0.392,  24.097,  285.537,   76.1, 191.2, -16.0,  11.8,  9.0,  0.9},
{ 4, 395, 386, 1652,   0.368,  12.813,  126.687,    1.6, 131.9,  10.7,   4.0,  8.5,  6.0},
{ 4, 396, 395, 1677,   0.240,  24.641,  145.414,    6.3, 142.5,   8.8,   5.4, 12.5, -9.8},
{ 4, 397, 396, 1677,   0.240,  24.641,  170.055,   11.7, 155.0,  -1.0,   5.4, 12.5, -9.8},
{ 4, 398, 397, 1677,   0.240,  24.641,  194.696,   17.2, 167.5, -10.8,   5.4, 12.5, -9.8},
{ 4, 399, 398, 1677,   0.240,  24.641,  219.337,   22.6, 180.0, -20.6,   5.4, 12.5, -9.8},
{ 4, 400, 399, 1677,   0.240,  24.641,  243.978,   28.0, 192.6, -30.4,   5.4, 12.5, -9.8},
{ 4, 401, 400, 1680,   0.321,  17.372,  264.984,   26.8, 202.7, -31.0,  -7.8,  7.7,  8.5},
{ 4, 402, 401, 1686,   0.240,  22.413,  284.877,   22.6, 213.9, -32.4,  -0.6, 14.9,-11.3},
{ 4, 403, 402, 1686,   0.240,  22.413,  307.291,   22.0, 228.8, -43.7,  -0.6, 14.9,-11.3},
{ 4, 404, 356, 1693,   0.880,  14.667,  110.079,   -3.3, 129.1,   0.7,   1.8, 13.7, -0.0},
{ 4, 405, 404, 1693,   0.880,  14.667,  124.745,   -1.5, 142.8,   0.7,   1.8, 13.7, -0.0},
{ 4, 406, 405, 1703,   0.400,  15.012,  139.585,    3.6, 153.5,   1.7,   8.5,  7.6,  2.0},
{ 4, 407, 406, 1703,   0.400,  15.012,  154.596,   12.1, 161.1,   3.7,   8.5,  7.6,  2.0},
{ 4, 408, 407, 1725,   0.240,  21.747,  172.976,   22.1, 166.4,   2.9,  11.6,  2.9, -3.5},
{ 4, 409, 408, 1725,   0.240,  21.747,  194.723,   33.7, 169.3,  -0.6,  11.6,  2.9, -3.5},
{ 4, 410, 409, 1725,   0.240,  21.747,  216.470,   45.3, 172.2,  -4.1,  11.6,  2.9, -3.5},
{ 4, 411, 410, 1725,   0.240,  21.747,  238.217,   56.9, 175.1,  -7.6,  11.6,  2.9, -3.5},
{ 4, 412, 411, 1725,   0.240,  21.747,  259.965,   68.5, 178.0, -11.1,  11.6,  2.9, -3.5},
{ 4, 413, 407, 1748,   0.400,  24.247,  174.226,   20.3, 171.5,   4.7,   8.0, 13.2,  0.1},
{ 4, 414, 413, 1748,   0.400,  24.247,  198.473,   28.3, 184.7,   4.8,   8.0, 13.2,  0.1},
{ 4, 415, 414, 1748,   0.400,  24.247,  222.720,   36.2, 197.9,   4.9,   8.0, 13.2,  0.1},
{ 4, 416, 415, 1781,   0.240,  23.225,  246.456,   44.0, 211.4,   1.6,   7.6, 13.8, -6.8},
{ 4, 417, 416, 1781,   0.240,  23.225,  269.681,   51.6, 225.2,  -5.2,   7.6, 13.8, -6.8},
{ 4, 418, 417, 1781,   0.240,  23.225,  292.906,   59.2, 239.0, -11.9,   7.6, 13.8, -6.8},
{ 4, 419, 418, 1781,   0.240,  23.225,  316.131,   66.8, 252.8, -18.7,   7.6, 13.8, -6.8},
{ 4, 420, 419, 1781,   0.240,  23.225,  339.356,   74.3, 266.6, -25.4,   7.6, 13.8, -6.8},
{ 4, 421, 415, 1802,   0.400,  20.755,  245.221,   41.7, 211.9,   6.8,   2.9, 14.7,  3.8},
{ 4, 422, 421, 1802,   0.400,  20.755,  265.976,   44.6, 226.6,  10.6,   2.9, 14.7,  3.8},
{ 4, 423, 422, 1802,   0.400,  20.755,  286.731,   47.5, 241.3,  14.3,   2.9, 14.7,  3.8},
{ 4, 424, 423, 1802,   0.400,  20.755,  307.485,   50.5, 256.0,  18.1,   2.9, 14.7,  3.8},
{ 4, 425, 424, 1802,   0.400,  20.755,  328.240,   53.4, 270.7,  21.8,   2.9, 14.7,  3.8},
{ 4, 426, 405, 1808,   0.720,  14.808,  139.483,   -0.2, 156.7,   0.7,   1.0, 14.0,  0.0},
{ 4, 427, 426, 1808,   0.720,  14.808,  154.291,    0.8, 170.7,   0.7,   1.0, 14.0,  0.0},
{ 4, 428, 427, 1810,   0.720,   0.890,  162.140,    1.3, 178.2,   0.7,   0.0,  0.9,  0.0},
{ 4, 429, 428, 1839,   0.240,  22.560,  173.865,    0.1, 184.9,   1.4,  -2.4, 12.5,  1.4},
{ 4, 430, 429, 1839,   0.240,  22.560,  196.425,   -2.4, 197.4,   2.8,  -2.4, 12.5,  1.4},
{ 4, 431, 430, 1839,   0.240,  22.560,  218.984,   -4.8, 209.8,   4.1,  -2.4, 12.5,  1.4},
{ 4, 432, 431, 1839,   0.240,  22.560,  241.544,   -7.2, 222.3,   5.5,  -2.4, 12.5,  1.4},
{ 4, 433, 432, 1839,   0.240,  22.560,  264.104,   -9.6, 234.8,   6.9,  -2.4, 12.5,  1.4},
{ 4, 434, 433, 1839,   0.240,  22.560,  286.663,  -12.1, 247.3,   8.3,  -2.4, 12.5,  1.4},
{ 4, 435, 434, 1839,   0.240,  22.560,  309.223,  -14.5, 259.8,   9.7,  -2.4, 12.5,  1.4},
{ 4, 436, 435, 1839,   0.240,  22.560,  331.782,  -16.9, 272.3,  11.0,  -2.4, 12.5,  1.4},
{ 4, 437, 436, 1839,   0.240,  22.560,  354.342,  -19.3, 284.8,  12.4,  -2.4, 12.5,  1.4},
{ 4, 438, 428, 1842,   0.720,   8.163,  166.667,    2.4, 182.4,   1.1,   2.3,  7.6,  0.8},
{ 4, 439, 438, 1849,   0.240,  21.590,  181.543,    0.6, 191.1,   7.4,  -5.9,  9.8, 12.0},
{ 4, 440, 438, 1854,   0.720,  22.892,  182.195,    5.9, 195.9,  -0.8,   4.7, 19.4, -4.5},
{ 4, 441, 440, 1895,   0.240,  24.626,  205.954,    9.1, 213.3,  -1.0,   1.6, 15.5,  4.1},
{ 4, 442, 441, 1895,   0.240,  24.626,  230.579,   10.7, 228.8,   3.1,   1.6, 15.5,  4.1},
{ 4, 443, 442, 1895,   0.240,  24.626,  255.205,   12.3, 244.3,   7.3,   1.6, 15.5,  4.1},
{ 4, 444, 443, 1895,   0.240,  24.626,  279.831,   13.9, 259.8,  11.4,   1.6, 15.5,  4.1},
{ 4, 445, 444, 1895,   0.240,  24.626,  304.457,   15.5, 275.3,  15.5,   1.6, 15.5,  4.1},
{ 4, 446, 445, 1895,   0.240,  24.626,  329.083,   17.1, 290.8,  19.6,   1.6, 15.5,  4.1},
{ 4, 447, 446, 1895,   0.240,  24.626,  353.709,   18.8, 306.3,  23.8,   1.6, 15.5,  4.1},
{ 4, 448, 447, 1895,   0.240,  24.626,  378.335,   20.4, 321.8,  27.9,   1.6, 15.5,  4.1},
{ 4, 449, 440, 1897,   0.720,   4.760,  196.021,    9.0, 206.8,  -4.7,   1.5,  2.4, -3.3},
{ 4, 450, 449, 1925,   0.240,  22.893,  209.847,    5.5, 215.1,  -4.1,  -8.4, 14.2,  4.4},
{ 4, 451, 450, 1925,   0.240,  22.893,  232.741,   -2.8, 229.2,   0.2,  -8.4, 14.2,  4.4},
{ 4, 452, 451, 1925,   0.240,  22.893,  255.634,  -11.2, 243.4,   4.6,  -8.4, 14.2,  4.4},
{ 4, 453, 452, 1925,   0.240,  22.893,  278.527,  -19.6, 257.6,   9.0,  -8.4, 14.2,  4.4},
{ 4, 454, 453, 1925,   0.240,  22.893,  301.421,  -28.0, 271.7,  13.3,  -8.4, 14.2,  4.4},
{ 4, 455, 454, 1925,   0.240,  22.893,  324.314,  -36.3, 285.9,  17.7,  -8.4, 14.2,  4.4},
{ 4, 456, 455, 1925,   0.240,  22.893,  347.208,  -44.7, 300.1,  22.0,  -8.4, 14.2,  4.4},
{ 4, 457, 449, 1936,   0.560,  15.675,  206.238,   11.2, 214.0,  -5.9,   2.9, 12.0,  0.8},
{ 4, 458, 457, 1936,   0.560,  15.675,  221.913,   14.0, 225.9,  -5.2,   2.9, 12.0,  0.8},
{ 4, 459, 458, 1956,   0.405,  21.020,  240.261,   17.9, 240.4,  -6.4,   5.0, 17.0, -3.1},
{ 4, 460, 459, 1956,   0.405,  21.020,  261.281,   22.9, 257.4,  -9.5,   5.0, 17.0, -3.1},
{ 4, 461, 460, 1956,   0.405,  21.020,  282.302,   27.9, 274.3, -12.6,   5.0, 17.0, -3.1},
{ 4, 462, 461, 1956,   0.405,  21.020,  303.322,   32.9, 291.3, -15.7,   5.0, 17.0, -3.1},
{ 4, 463, 462, 1964,   0.400,  25.792,  326.729,   42.4, 305.2, -22.3,  14.0, 10.9,-10.0},
{ 4, 464, 463, 2002,   0.242,  24.561,  351.905,   53.2, 317.6, -26.2,   7.7, 13.8,  2.2},
{ 4, 465, 464, 2002,   0.242,  24.561,  376.465,   60.9, 331.4, -24.0,   7.7, 13.8,  2.2},
{ 4, 466, 465, 2002,   0.242,  24.561,  401.026,   68.6, 345.2, -21.8,   7.7, 13.8,  2.2},
{ 4, 467, 466, 2002,   0.242,  24.561,  425.587,   76.3, 359.0, -19.6,   7.7, 13.8,  2.2},
{ 4, 468, 467, 2002,   0.242,  24.561,  450.147,   84.0, 372.8, -17.4,   7.7, 13.8,  2.2},
{ 4, 469, 468, 2002,   0.242,  24.561,  474.708,   91.6, 386.6, -15.2,   7.7, 13.8,  2.2},
{ 4, 470, 469, 2019,   0.240,  23.374,  498.676,   97.8, 398.7, -17.0,   4.6, 10.4, -6.0},
{ 4, 471, 470, 2019,   0.240,  23.374,  522.050,  102.4, 409.1, -23.0,   4.6, 10.4, -6.0},
{ 4, 472, 471, 2019,   0.240,  23.374,  545.424,  107.0, 419.5, -29.0,   4.6, 10.4, -6.0},
{ 4, 473, 469, 2048,   0.240,  22.841,  498.409,   96.2, 402.1, -14.6,   1.4, 17.1, -1.2},
{ 4, 474, 473, 2048,   0.240,  22.841,  521.249,   97.6, 419.3, -15.8,   1.4, 17.1, -1.2},
{ 4, 475, 474, 2048,   0.240,  22.841,  544.090,   99.0, 436.4, -17.0,   1.4, 17.1, -1.2},
{ 4, 476, 475, 2048,   0.240,  22.841,  566.930,  100.4, 453.6, -18.3,   1.4, 17.1, -1.2},
{ 4, 477, 476, 2048,   0.240,  22.841,  589.771,  101.8, 470.7, -19.5,   1.4, 17.1, -1.2},
{ 4, 478, 462, 2062,   0.400,  20.188,  323.926,   34.2, 308.0, -19.9,  -2.4, 16.5, -5.1},
{ 4, 479, 478, 2062,   0.400,  20.188,  344.114,   31.8, 324.5, -25.0,  -2.4, 16.5, -5.1},
{ 4, 480, 479, 2062,   0.400,  20.188,  364.302,   29.4, 341.0, -30.1,  -2.4, 16.5, -5.1},
{ 4, 481, 480, 2062,   0.400,  20.188,  384.489,   27.0, 357.5, -35.2,  -2.4, 16.5, -5.1},
{ 4, 482, 481, 2102,   0.240,  22.212,  405.689,   24.8, 374.0, -38.5,  -2.0, 16.6, -1.4},
{ 4, 483, 482, 2102,   0.240,  22.212,  427.901,   22.9, 390.5, -39.9,  -2.0, 16.6, -1.4},
{ 4, 484, 483, 2102,   0.240,  22.212,  450.112,   20.9, 407.1, -41.3,  -2.0, 16.6, -1.4},
{ 4, 485, 484, 2102,   0.240,  22.212,  472.324,   18.9, 423.7, -42.7,  -2.0, 16.6, -1.4},
{ 4, 486, 485, 2102,   0.240,  22.212,  494.536,   16.9, 440.2, -44.1,  -2.0, 16.6, -1.4},
{ 4, 487, 486, 2102,   0.240,  22.212,  516.748,   14.9, 456.8, -45.5,  -2.0, 16.6, -1.4},
{ 4, 488, 487, 2102,   0.240,  22.212,  538.959,   12.9, 473.3, -46.9,  -2.0, 16.6, -1.4},
{ 4, 489, 488, 2102,   0.240,  22.212,  561.171,   10.9, 489.9, -48.3,  -2.0, 16.6, -1.4},
{ 4, 490, 481, 2146,   0.240,  22.477,  405.821,   22.2, 373.3, -34.3,  -7.2, 15.2,  6.9},
{ 4, 491, 490, 2146,   0.240,  22.477,  428.298,   15.1, 388.5, -27.4,  -7.2, 15.2,  6.9},
{ 4, 492, 491, 2146,   0.240,  22.477,  450.775,    7.9, 403.8, -20.5,  -7.2, 15.2,  6.9},
{ 4, 493, 492, 2146,   0.240,  22.477,  473.252,    0.7, 419.0, -13.5,  -7.2, 15.2,  6.9},
{ 4, 494, 493, 2146,   0.240,  22.477,  495.729,   -6.4, 434.2,  -6.6,  -7.2, 15.2,  6.9},
{ 4, 495, 494, 2146,   0.240,  22.477,  518.206,  -13.6, 449.5,   0.4,  -7.2, 15.2,  6.9},
{ 4, 496, 495, 2146,   0.240,  22.477,  540.683,  -20.8, 464.7,   7.3,  -7.2, 15.2,  6.9},
{ 4, 497, 496, 2146,   0.240,  22.477,  563.160,  -28.0, 479.9,  14.2,  -7.2, 15.2,  6.9},
{ 4, 498, 427, 2149,   0.345,   6.761,  165.075,   -0.7, 179.2,  17.4,  -3.9,  2.9, 33.3},
{ 4, 499, 498, 2157,   0.400,  15.011,  175.961,   -1.5, 185.5,  35.4,   2.2,  9.8,  2.8},
{ 4, 500, 499, 2157,   0.400,  15.011,  190.973,    0.7, 195.3,  38.1,   2.2,  9.8,  2.8},
{ 4, 501, 500, 2177,   0.249,  22.383,  209.670,    3.9, 207.1,  42.5,   4.3, 13.7,  6.1},
{ 4, 502, 501, 2177,   0.249,  22.383,  232.053,    8.2, 220.8,  48.6,   4.3, 13.7,  6.1},
{ 4, 503, 502, 2177,   0.249,  22.383,  254.436,   12.5, 234.5,  54.7,   4.3, 13.7,  6.1},
{ 4, 504, 503, 2177,   0.249,  22.383,  276.819,   16.7, 248.2,  60.8,   4.3, 13.7,  6.1},
{ 4, 505, 504, 2177,   0.249,  22.383,  299.203,   21.0, 262.0,  66.9,   4.3, 13.7,  6.1},
{ 4, 506, 505, 2177,   0.249,  22.383,  321.586,   25.3, 275.7,  73.0,   4.3, 13.7,  6.1},
{ 3, 507,   0, 2183,   1.040,  16.544,    8.272,   -3.2,  -6.4,  -6.5,  -4.2, -3.7,-15.0},
{ 3, 508, 507, 2183,   1.040,  16.544,   24.816,   -7.4, -10.2, -21.5,  -4.2, -3.7,-15.0},
{ 3, 509, 508, 2200,   0.599,  23.593,   44.885,  -13.6, -17.4, -25.4,  -8.2,-10.8,  7.2},
{ 3, 510, 509, 2200,   0.599,  23.593,   68.478,  -21.8, -28.2, -18.2,  -8.2,-10.8,  7.2},
{ 3, 511, 510, 2200,   0.599,  23.593,   92.071,  -29.9, -39.0, -11.0,  -8.2,-10.8,  7.2},
{ 3, 512, 511, 2200,   0.599,  23.593,  115.665,  -38.1, -49.8,  -3.8,  -8.2,-10.8,  7.2},
{ 3, 513, 512, 2204,   0.432,  20.241,  137.582,  -44.8, -57.4,   0.8,  -5.3, -4.6,  2.0},
{ 3, 514, 513, 2204,   0.432,  20.241,  157.823,  -50.1, -62.0,   2.8,  -5.3, -4.6,  2.0},
{ 3, 515, 514, 2213,   0.240,  22.080,  178.983,  -54.8, -69.4,   2.2,  -4.0,-10.3, -3.2},
{ 3, 516, 515, 2213,   0.240,  22.080,  201.064,  -58.8, -79.7,  -1.0,  -4.0,-10.3, -3.2},
{ 3, 517, 516, 2213,   0.240,  22.080,  223.144,  -62.8, -90.0,  -4.2,  -4.0,-10.3, -3.2},
{ 3, 518, 517, 2245,   0.399,  23.684,  246.026,  -65.0,-105.6,  -4.4,  -0.4,-20.9,  2.8},
{ 3, 519, 518, 2245,   0.399,  23.684,  269.711,  -65.4,-126.5,  -1.6,  -0.4,-20.9,  2.8},
{ 3, 520, 519, 2245,   0.399,  23.684,  293.395,  -65.8,-147.3,   1.1,  -0.4,-20.9,  2.8},
{ 3, 521, 520, 2245,   0.399,  23.684,  317.079,  -66.2,-168.2,   3.9,  -0.4,-20.9,  2.8},
{ 3, 522, 521, 2245,   0.399,  23.684,  340.764,  -66.6,-189.1,   6.6,  -0.4,-20.9,  2.8},
{ 3, 523, 522, 2245,   0.399,  23.684,  364.448,  -67.0,-210.0,   9.4,  -0.4,-20.9,  2.8},
{ 3, 524, 514, 2256,   0.240,  25.619,  180.753,  -61.2, -70.8,   2.3, -16.8,-12.9, -3.0},
{ 3, 525, 524, 2256,   0.240,  25.619,  206.371,  -78.0, -83.7,  -0.8, -16.8,-12.9, -3.0},
{ 3, 526, 525, 2276,   0.399,  21.398,  229.880,  -92.1, -96.6,  -2.0, -11.3,-12.9,  0.4},
{ 3, 527, 526, 2276,   0.399,  21.398,  251.278, -103.4,-109.5,  -1.6, -11.3,-12.9,  0.4},
{ 3, 528, 527, 2276,   0.399,  21.398,  272.677, -114.7,-122.3,  -1.2, -11.3,-12.9,  0.4},
{ 3, 529, 528, 2276,   0.399,  21.398,  294.075, -126.0,-135.2,  -0.8, -11.3,-12.9,  0.4},
{ 3, 530, 529, 2276,   0.399,  21.398,  315.473, -137.2,-148.0,  -0.4, -11.3,-12.9,  0.4},
{ 3, 531, 530, 2276,   0.399,  21.398,  336.871, -148.5,-160.9,   0.0, -11.3,-12.9,  0.4},
{ 3, 532,   0, 2280,   0.720,  13.930,    6.965,   -0.5, -11.0,  -3.5,   2.0, -6.8, -9.0},
{ 3, 533, 532, 2386,   0.720,  20.541,   24.200,    4.8, -20.6,  -9.4,   8.7,-12.3, -2.8},
{ 3, 534, 533, 2386,   0.720,  20.541,   44.741,   13.5, -32.9, -12.1,   8.7,-12.3, -2.8},
{ 3, 535, 534, 2390,   0.720,  16.394,   63.209,   21.0, -45.3, -11.5,   6.3,-12.6,  4.0},
{ 3, 536, 535, 2393,   0.400,  14.555,   78.684,   28.2, -51.3, -14.8,   8.0,  0.7,-10.5},
{ 3, 537, 536, 2407,   0.240,  20.555,   96.239,   39.2, -52.0, -22.5,  14.0, -2.1, -4.9},
{ 3, 538, 537, 2407,   0.240,  20.555,  116.794,   53.2, -54.1, -27.4,  14.0, -2.1, -4.9},
{ 3, 539, 538, 2407,   0.240,  20.555,  137.349,   67.2, -56.1, -32.3,  14.0, -2.1, -4.9},
{ 3, 540, 539, 2407,   0.240,  20.555,  157.905,   81.2, -58.2, -37.3,  14.0, -2.1, -4.9},
{ 3, 541, 536, 2416,   0.560,  25.606,   98.764,   36.6, -58.1, -19.1,   8.7,-14.3,  1.8},
{ 3, 542, 541, 2416,   0.560,  25.606,  124.370,   45.3, -72.4, -17.4,   8.7,-14.3,  1.8},
{ 3, 543, 542, 2423,   0.560,  23.432,  148.888,   55.7, -80.7, -18.1,  12.2, -2.4, -3.1},
{ 3, 544, 543, 2423,   0.560,  23.432,  172.320,   67.9, -83.1, -21.2,  12.2, -2.4, -3.1},
{ 3, 545, 544, 2427,   0.410,  22.050,  195.061,   83.1, -88.3, -22.8,  18.2, -8.0,  0.0},
{ 3, 546, 545, 2447,   0.400,  25.135,  218.653,   98.9, -94.6, -25.0,  13.5, -4.5, -4.6},
{ 3, 547, 546, 2447,   0.400,  25.135,  243.789,  112.4, -99.1, -29.6,  13.5, -4.5, -4.6},
{ 3, 548, 547, 2447,   0.400,  25.135,  268.924,  125.9,-103.6, -34.2,  13.5, -4.5, -4.6},
{ 3, 549, 545, 2449,   0.400,   9.217,  210.694,   95.8, -94.8, -24.1,   7.2, -5.0, -2.8},
{ 3, 550, 549, 2467,   0.247,  21.696,  226.151,  103.9, -99.4, -25.2,   9.0, -4.2,  0.6},
{ 3, 551, 550, 2467,   0.247,  21.696,  247.847,  112.9,-103.6, -24.6,   9.0, -4.2,  0.6},
{ 3, 552, 551, 2467,   0.247,  21.696,  269.543,  121.9,-107.7, -24.0,   9.0, -4.2,  0.6},
{ 3, 553, 552, 2467,   0.247,  21.696,  291.239,  130.9,-111.9, -23.5,   9.0, -4.2,  0.6},
{ 3, 554, 553, 2467,   0.247,  21.696,  312.935,  139.9,-116.1, -22.9,   9.0, -4.2,  0.6},
{ 3, 555, 554, 2467,   0.247,  21.696,  334.631,  148.9,-120.2, -22.3,   9.0, -4.2,  0.6},
{ 3, 556, 542, 2472,   0.560,  16.346,  145.346,   51.7, -85.0, -13.8,   4.2,-11.0,  5.4},
{ 3, 557, 556, 2472,   0.560,  16.346,  161.692,   56.0, -96.0,  -8.4,   4.2,-11.0,  5.4},
{ 3, 558, 557, 2494,   0.405,  22.106,  180.918,   63.6,-109.7,  -7.1,  10.9,-16.5, -2.8},
{ 3, 559, 558, 2494,   0.405,  22.106,  203.024,   74.5,-126.2,  -9.9,  10.9,-16.5, -2.8},
{ 3, 560, 559, 2494,   0.405,  22.106,  225.130,   85.4,-142.7, -12.7,  10.9,-16.5, -2.8},
{ 3, 561, 560, 2494,   0.405,  22.106,  247.236,   96.3,-159.2, -15.5,  10.9,-16.5, -2.8},
{ 3, 562, 561, 2494,   0.405,  22.106,  269.342,  107.2,-175.7, -18.3,  10.9,-16.5, -2.8},
{ 3, 563, 562, 2494,   0.405,  22.106,  291.448,  118.2,-192.2, -21.1,  10.9,-16.5, -2.8},
{ 3, 564, 563, 2494,   0.405,  22.106,  313.554,  129.1,-208.7, -23.9,  10.9,-16.5, -2.8},
{ 3, 565, 535, 2505,   0.569,  16.937,   79.875,   25.6, -57.8,  -8.6,   2.8,-12.4,  1.8},
{ 3, 566, 565, 2505,   0.569,  16.937,   96.812,   28.3, -70.2,  -6.9,   2.8,-12.4,  1.8},
{ 3, 567, 566, 2510,   0.400,  22.399,  116.481,   36.7, -84.0,  -7.6,  14.0,-15.2, -3.3},
{ 3, 568, 567, 2537,   0.244,  25.086,  140.223,   47.1, -99.0, -10.6,   6.9,-14.7, -2.8},
{ 3, 569, 568, 2537,   0.244,  25.086,  165.309,   54.1,-113.7, -13.4,   6.9,-14.7, -2.8},
{ 3, 570, 569, 2537,   0.244,  25.086,  190.395,   61.0,-128.4, -16.1,   6.9,-14.7, -2.8},
{ 3, 571, 570, 2537,   0.244,  25.086,  215.480,   67.9,-143.1, -18.9,   6.9,-14.7, -2.8},
{ 3, 572, 571, 2537,   0.244,  25.086,  240.566,   74.8,-157.7, -21.6,   6.9,-14.7, -2.8},
{ 3, 573, 572, 2537,   0.244,  25.086,  265.651,   81.7,-172.4, -24.4,   6.9,-14.7, -2.8},
{ 3, 574, 573, 2537,   0.244,  25.086,  290.737,   88.6,-187.1, -27.1,   6.9,-14.7, -2.8},
{ 3, 575, 574, 2537,   0.244,  25.086,  315.823,   95.5,-201.8, -29.9,   6.9,-14.7, -2.8},
{ 3, 576, 566, 2541,   0.443,   7.787,  109.175,   30.9, -79.3,  -4.8,   2.3, -5.7,  2.5},
{ 3, 577, 576, 2575,   0.400,  25.163,  125.650,   32.0, -92.5,  -2.3,  -0.0,-20.7,  2.4},
{ 3, 578, 577, 2575,   0.400,  25.163,  150.813,   32.0,-113.2,   0.0,  -0.0,-20.7,  2.4},
{ 3, 579, 578, 2575,   0.400,  25.163,  175.976,   32.0,-133.9,   2.4,  -0.0,-20.7,  2.4},
{ 3, 580, 579, 2575,   0.400,  25.163,  201.140,   31.9,-154.6,   4.8,  -0.0,-20.7,  2.4},
{ 3, 581, 580, 2575,   0.400,  25.163,  226.303,   31.9,-175.4,   7.1,  -0.0,-20.7,  2.4},
{ 3, 582, 581, 2575,   0.400,  25.163,  251.467,   31.9,-196.1,   9.5,  -0.0,-20.7,  2.4},
{ 3, 583, 582, 2575,   0.400,  25.163,  276.630,   31.9,-216.8,  11.8,  -0.0,-20.7,  2.4},
{ 3, 584, 534, 2578,   0.560,  12.547,   61.286,   22.8, -41.8, -11.9,   9.9, -5.7,  3.3},
{ 3, 585, 584, 2586,   0.426,  23.560,   79.339,   34.6, -48.2, -14.9,  13.6, -7.1, -9.3},
{ 3, 586, 585, 2586,   0.426,  23.560,  102.899,   48.1, -55.3, -24.1,  13.6, -7.1, -9.3},
{ 3, 587, 586, 2590,   0.552,  18.738,  124.049,   56.7, -59.3, -27.2,   3.5, -1.1,  3.1},
{ 3, 588, 587, 2590,   0.552,  18.738,  142.787,   60.2, -60.4, -24.1,   3.5, -1.1,  3.1},
{ 3, 589, 588, 2590,   0.552,  18.738,  161.525,   63.7, -61.5, -21.0,   3.5, -1.1,  3.1},
{ 3, 590, 589, 2593,   0.560,  22.805,  182.296,   71.9, -66.1, -24.4,  12.9, -8.0, -9.8},
{ 3, 591, 590, 2619,   0.400,  24.831,  206.114,   84.2, -73.8, -33.0,  11.7, -7.5, -7.6},
{ 3, 592, 591, 2619,   0.400,  24.831,  230.945,   95.9, -81.3, -40.6,  11.7, -7.5, -7.6},
{ 3, 593, 592, 2619,   0.400,  24.831,  255.776,  107.5, -88.8, -48.2,  11.7, -7.5, -7.6},
{ 3, 594, 593, 2619,   0.400,  24.831,  280.607,  119.2, -96.2, -55.8,  11.7, -7.5, -7.6},
{ 3, 595, 594, 2619,   0.400,  24.831,  305.438,  130.9,-103.7, -63.4,  11.7, -7.5, -7.6},
{ 3, 596, 595, 2619,   0.400,  24.831,  330.269,  142.6,-111.2, -71.0,  11.7, -7.5, -7.6},
{ 3, 597, 596, 2619,   0.400,  24.831,  355.100,  154.2,-118.6, -78.6,  11.7, -7.5, -7.6},
{ 3, 598, 597, 2619,   0.400,  24.831,  379.931,  165.9,-126.1, -86.2,  11.7, -7.5, -7.6},
{ 3, 599, 598, 2619,   0.400,  24.831,  404.762,  177.6,-133.6, -93.8,  11.7, -7.5, -7.6},
{ 3, 600, 599, 2619,   0.400,  24.831,  429.592,  189.3,-141.0,-101.4,  11.7, -7.5, -7.6},
{ 3, 601, 590, 2621,   0.483,   5.876,  196.637,   77.9, -71.6, -31.6,  -0.8, -3.1, -4.8},
{ 3, 602, 601, 2636,   0.400,  19.860,  209.505,   82.8, -75.9, -39.3,  10.6, -5.4,-10.6},
{ 3, 603, 602, 2636,   0.400,  19.860,  229.365,   93.4, -81.3, -49.8,  10.6, -5.4,-10.6},
{ 3, 604, 603, 2636,   0.400,  19.860,  249.226,  104.0, -86.7, -60.4,  10.6, -5.4,-10.6},
{ 3, 605, 604, 2636,   0.400,  19.860,  269.086,  114.6, -92.2, -71.0,  10.6, -5.4,-10.6},
{ 3, 606, 605, 2650,   0.250,  22.868,  290.450,  127.5, -99.2, -80.4,  15.2, -8.6, -8.4},
{ 3, 607, 606, 2650,   0.250,  22.868,  313.319,  142.7,-107.9, -88.8,  15.2, -8.6, -8.4},
{ 3, 608, 607, 2650,   0.250,  22.868,  336.187,  157.8,-116.5, -97.2,  15.2, -8.6, -8.4},
{ 3, 609, 608, 2650,   0.250,  22.868,  359.056,  173.0,-125.2,-105.6,  15.2, -8.6, -8.4},
{ 3, 610, 589, 2663,   0.400,  22.168,  181.978,   71.7, -59.4, -28.6,  12.3,  5.2,-18.3},
{ 3, 611, 610, 2663,   0.400,  22.168,  204.146,   84.0, -54.3, -46.9,  12.3,  5.2,-18.3},
{ 3, 612, 611, 2677,   0.400,  23.820,  227.140,   96.3, -49.6, -54.5,  12.2,  4.2,  3.0},
{ 3, 613, 612, 2677,   0.400,  23.820,  250.960,  108.5, -45.4, -51.5,  12.2,  4.2,  3.0},
{ 3, 614, 613, 2677,   0.400,  23.820,  274.780,  120.8, -41.2, -48.5,  12.2,  4.2,  3.0},
{ 3, 615, 614, 2677,   0.400,  23.820,  298.600,  133.0, -37.0, -45.5,  12.2,  4.2,  3.0},
{ 3, 616, 611, 2685,   0.400,  21.403,  225.931,   92.6, -50.1, -56.8,   4.8,  3.0, -1.7},
{ 3, 617, 616, 2685,   0.400,  21.403,  247.333,   97.4, -47.1, -58.5,   4.8,  3.0, -1.7},
{ 3, 618, 617, 2685,   0.400,  21.403,  268.736,  102.2, -44.1, -60.2,   4.8,  3.0, -1.7},
{ 3, 619, 618, 2685,   0.400,  21.403,  290.139,  107.0, -41.1, -61.9,   4.8,  3.0, -1.7},

// end marker
{-1,  -1,  -1,   -1,   0.000,   0.000,    0.000,    0.0,   0.0,   0.0,   0.0,  0.0,  0.0}

};	// end of table

return cellMorphology;

}	// end of function
