/*
 * Copyright (C) 2004 Evan Thomas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*************************************************
  This file implements the Cell class for Python.
**************************************************/

#define P3_MODULE
#include "ndl.h"

static int clear(Cell *self) {
    il_clear(self->compartments);
    Py_XDECREF(self->compartments);
    self->compartments = 0;
    return 0;
}

static int traverse(Cell *self, visitproc v, void *a) {
    int i;
    if( !self->compartments ) return 0;
    for(i=0; i<il_length(self->compartments); i++)
        if( v(il_get(self->compartments, i), a) < 0 ) return -1;
    return 0;
}

static void del(Cell *self) {
    Py_XDECREF(self->Y);
    Py_XDECREF(self->DYDT);
    Py_XDECREF(self->DYDT_copy);
    Py_XDECREF(self->Y_copy);
    Py_XDECREF(self->Yinterp);

    clear(self);

    PyMem_Free(self->mustStep);

    /* Release Dynamics and Synapse objects */
    self->ob_type->tp_free((PyObject*)self);
}

static int init(Cell *self, PyObject *args) {
    double_array *x;
    internal_list *l;
    int i;
    if( (x=da_new()) == NULL ) return -1; else self->Y = x; 
    if( (x=da_new()) == NULL ) return -1; else self->DYDT = x; 
    if( (x=da_new()) == NULL ) return -1; else self->DYDT_copy = x; 
    if( (x=da_new()) == NULL ) return -1; else self->Y_copy = x; 
    if( (x=da_new()) == NULL ) return -1; else self->Yinterp = x;
    if( (x=da_new()) == NULL ) return -1; else self->fac = x;
    if( (x=da_new()) == NULL ) return -1; else self->fac_copy = x;

    if( (l=il_new("Compartment")) == NULL ) return -1;
    else self->compartments = l;

    /* Must step buffers */
    self->mustStep      = PyMem_Malloc(MUSTSTEP_BUFSIZE*sizeof(*self->mustStep));
    if( !self->mustStep ) {
        PyErr_NoMemory();
        return -1;
    }
    self->mustStepLen = MUSTSTEP_BUFSIZE;
    for(i=0; i<MUSTSTEP_BUFSIZE; i++)
        self->mustStep[i].time = MUSTSTEP_SENTINEL;

	self->mpi_rank = 0;
	self->isLocal = mpi_rank==0;

    return 0;
}

int addState(Cell *self, int n) {
    int i, rc=self->Y->len;
    for(i=0; i<n; i++) {
        da_append(self->Y, 0);
        da_append(self->DYDT, 0);
        da_append(self->Y_copy, 0);
        da_append(self->DYDT_copy, 0);
        da_append(self->Yinterp, 0);
        da_append(self->fac, 0);
        da_append(self->fac_copy, 0);
    }
    return rc;
}

void remState(Cell *self, int n) {
	da_slice_del(self->Y, n, n);
	da_slice_del(self->DYDT, n, n);
	da_slice_del(self->Y_copy, n, n);
	da_slice_del(self->DYDT_copy, n, n);
	da_slice_del(self->Yinterp, n, n);
	da_slice_del(self->fac, n, n);
	da_slice_del(self->fac_copy, n, n);
}

static PyObject *repr(Cell *self) {
    return PyString_FromFormat("C Cell object at %p, refcnt=%d",
        self, ((PyObject*)self)->ob_refcnt);
}

static PyMethodDef methods[] = {
 	{NULL, NULL, 0, NULL}		/* sentinel */
};

static PyMemberDef members[] = {
    {"timeSample", T_DOUBLE, offsetof(Cell, timeSample),
     0, "tracing sample interval"},

    {"time", T_DOUBLE, offsetof(Cell, time), 
     READONLY, "current time point of cell"},

    {"step", T_DOUBLE, offsetof(Cell, step), 
     0, "anticipated next integration step size"},

    {"laststepaccept", T_INT, offsetof(Cell, laststepaccept), 
     READONLY, "True is the last integration step on this cell was accepted"},

    {"id", T_INT, offsetof(Cell, id), 
     READONLY, "unique id within partition"},

    {"isLocal", T_INT, offsetof(Cell, isLocal), 
     RO, "true if the cell is being solved in this node"},

    {"stepAccepts", T_INT, offsetof(Cell, stepAccepts),
    READONLY, "number of successful steps"},

    {"stepTotal", T_INT, offsetof(Cell, stepTotal),
    READONLY, "total number of steps"},

    {"functionCnt", T_INT, offsetof(Cell, functionCnt),
    READONLY, "total number of function evaluations"},

    {"jacobianCnt", T_INT, offsetof(Cell, jacobianCnt),
    READONLY, "total number of Jacobian evaluations (implicit methods)"},

    {"maxStepCnt", T_INT, offsetof(Cell, maxStepCnt),
    READONLY, "number of times maxStep was encounted"},

    {"badStepCnt", T_INT, offsetof(Cell, badStepCnt),
    READONLY, "total number of Jacobian evaluations (implicit methods)"},

    {"newtonCnt", T_INT, offsetof(Cell, newtonCnt),
    READONLY, "total number of Newton interations (implicit methods)"},

    {"compartments", T_OBJECT_EX, offsetof(Cell, compartments), 
     READONLY, "list of Comparment objects attached to this cell"},

    {"Y", T_OBJECT_EX, offsetof(Cell, Y), 
     READONLY, "the state vector"},

    {"DYDT", T_OBJECT_EX, offsetof(Cell, DYDT), 
     READONLY, "the derivative of the state vector"},    

    {"stepTrace", T_INT, offsetof(Cell, stepTrace),
     0, "mark cell for step level tracing tracing"}, 

    {"sparseLinearAlgebra", T_INT, offsetof(Cell, sparseLinearAlgebra),
     0, "use sparse matrices for linear algebra in implicit methods"},

 	{NULL}		/* sentinel */
};

static PyObject *getMethod(Cell *self) {
    char *s = "--- unset ---";
    if( self->solver == rk32_solve )   s = "rk32";
    if( self->solver == rw23_solve )   s = "rw23";
    if( self->solver == rd5_solve )    s = "rdIIA";
    if( self->solver == hines_solve )  s = "hines";
    if( self->solver == hines_fixed_step_solve )  s = "hines_fixed_step_";
    return PyString_FromString(s);
}

static int setMethod(Cell *self, PyObject *mpy) {
    char *m = PyString_AsString(mpy);

    if( !m ) return -1;

    if( strcmp(m, rk32)==0 ) {
        self->solver = rk32_solve;
        self->interpolator = rk32_interp;
		self->solverInit = rk32_init;
        return 0;
    }

    if( strcmp(m, rdIIA)==0 ) {
        self->solver = rd5_solve;
        self->interpolator = rd5_interp;
		self->solverInit = rd5_init;
        return 0;
    }

    if( strcmp(m, rw23)==0 ) {
        self->solver = rw23_solve;
        self->interpolator = rw23_interp;
		self->solverInit = rw23_init;
        return 0;
    }

    if( strcmp(m, hines)==0 ) {
        self->solver = hines_solve;
        self->interpolator = hines_interp;
		self->solverInit = hines_init;
        return 0;
    }

    if( strcmp(m, hines_fixed_step)==0 ) {
        self->solver = hines_fixed_step_solve;
        self->interpolator = hines_interp;
		self->solverInit = hines_init;
        return 0;
    }

    PyErr_SetString(PyExc_TypeError, 
        "unknown method");

    return -1;
}


static int setMPIrank(Cell *self, PyObject *rankpy) {
	int r;

    if( !PyInt_Check(rankpy) ) {    
		PyErr_SetString(PyExc_TypeError, 
        "mpi_rank must be an integer");
        return -1;
    }

	r = PyInt_AsLong(rankpy);
	if( r<mpi_size && r>= 0) {
		self->mpi_rank = r;
		self->isLocal = r==mpi_rank;
		return 0;
	}
    PyErr_SetString(PyExc_TypeError, "mpi_rank must be less than mpi_size and greated than zero");
    return -1;
}

static PyObject *getMPIrank(Cell *self) {
    return PyLong_FromLong(self->mpi_rank);
}

static PyGetSetDef getseters[] = {
    {"method", (getter)getMethod, (setter)setMethod,
	"numerical ODE method", NULL},
    {"mpi_rank",  (getter)getMPIrank, (setter)setMPIrank,
	"mpi rank of the process solving this cell", NULL},
    {NULL}  /* Sentinel */
};

PyTypeObject p3_CellType = {
    PyObject_HEAD_INIT(NULL)
    0,                         /*ob_size*/
    "_p3.Cell",             /*tp_name*/
    sizeof(Cell),      /*tp_basicsize*/
    0,                         /*tp_itemsize*/
    (destructor)del,                         /*tp_dealloc*/
    0,                         /*tp_print*/
    0,                         /*tp_getattr*/
    0,                         /*tp_setattr*/
    0,                         /*tp_compare*/
    (reprfunc)repr,                         /*tp_repr*/
    0,                         /*tp_as_number*/
    0,                         /*tp_as_sequence*/
    0,                         /*tp_as_mapping*/
    0,                         /*tp_hash */
    0,                         /*tp_call*/
    0,                         /*tp_str*/
    0,                         /*tp_getattro*/
    0,                         /*tp_setattro*/
    0,                         /*tp_as_buffer*/
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE | Py_TPFLAGS_HAVE_GC,        /*tp_flags*/
    "Parplex Cell object",       /* tp_doc */
    (traverseproc)traverse,		 /* tp_traverse */
    (inquiry)clear,		         /* tp_clear */
    0,		               /* tp_richcompare */
    0,		               /* tp_weaklistoffset */
    0,		               /* tp_iter */
    0,		               /* tp_iternext */
    methods,               /* tp_methods */
    members,               /* tp_members */
    getseters,                         /* tp_getset */
    0,                         /* tp_base */
    0,                         /* tp_dict */
    0,                         /* tp_descr_get */
    0,                         /* tp_descr_set */
    0,                         /* tp_dictoffset */
    (initproc)init,      /* tp_init */
    0,                         /* tp_alloc */
    PyType_GenericNew                 /* tp_new */
};
